/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.eig;

import mt.Vector;
import mt.Vector.Norm;

/**
 * Partial implementation of EigenvalueIterationMonitor
 */
public abstract class AbstractEigenvalueIterationMonitor
	implements EigenvalueIterationMonitor {

	/**
	 * Iteration number
	 */
	protected int iter;

	/**
	 * Eigenvalue iteration reporter
	 */
	protected EigenvalueIterationReporter reporter;

	/**
	 * Current residual
	 */
	protected double r;

	/**
	 * Vector-norm
	 */
	protected Norm norm;

	/**
	 * Constructor for AbstractEigenvalueIterationMonitor. Performs no
	 * iteration reporting
	 */
	public AbstractEigenvalueIterationMonitor() {
		reporter = new NoEigenvalueIterationReporter();
	}

	public void setFirst() {
		iter = 0;
	}

	public boolean isFirst() {
		return iter == 0;
	}

	public void next() {
		iter++;
	}

	public int iterations() {
		return iter;
	}

	public double residual() {
		return r;
	}

	public boolean converged(Vector r, double v, Vector x)
		throws IterativeEigenvalueSolverNotConvergedException {
		return converged(r.norm(norm), v, x);
	}

	public boolean converged(Vector r, double v)
		throws IterativeEigenvalueSolverNotConvergedException {
		return converged(r.norm(norm), v);
	}

	public boolean converged(Vector r)
		throws IterativeEigenvalueSolverNotConvergedException {
		return converged(r.norm(norm));
	}

	public boolean converged(double r, double v, Vector x)
		throws IterativeEigenvalueSolverNotConvergedException {
		reporter.monitor(r, v, x, iter);
		this.r = r;
		return convergedI(r);
	}

	public boolean converged(double r, double v)
		throws IterativeEigenvalueSolverNotConvergedException {
		reporter.monitor(r, v, iter);
		this.r = r;
		return convergedI(r);
	}

	public boolean converged(double r)
		throws IterativeEigenvalueSolverNotConvergedException {
		reporter.monitor(r, iter);
		this.r = r;
		return convergedI(r);
	}

	public void setIterationReporter(EigenvalueIterationReporter reporter) {
		this.reporter = reporter;
	}

	public EigenvalueIterationReporter getIterationReporter() {
		return reporter;
	}

	/**
	 * Returns true if the iteration has converged
	 */
	protected abstract boolean convergedI(double r)
		throws IterativeEigenvalueSolverNotConvergedException;

	public Norm getNormType() {
		return norm;
	}

	public void setNormType(Norm norm) {
		this.norm= norm;
	}

}
