/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
/*
 * Portions copyright  1999 CERN - European Organization for Nuclear Research.
 * 
 * Permission to use, copy, modify, distribute and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation. CERN
 * makes no representations about the suitability of this software for any
 * purpose. It is provided "as is" without expressed or implied warranty.
 */

package smt;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Iterator;

import mt.AbstractVector;
import mt.DenseVector;
import mt.Vector;
import mt.VectorEntry;

/**
 * Sparse vector stored in a hash table
 */
public class HashVector extends AbstractVector implements Serializable {

	private static final long serialVersionUID = -807267699591917119L;

	/**
	 * Long list of useful primes. From CERN
	 */
	private static final int[] primeCapacities = {
		//chunk #0
		Integer.MAX_VALUE,

		//chunk #1
		5,
			11,
			23,
			47,
			97,
			197,
			397,
			797,
			1597,
			3203,
			6421,
			12853,
			25717,
			51437,
			102877,
			205759,
			411527,
			823117,
			1646237,
			3292489,
			6584983,
			13169977,
			26339969,
			52679969,
			105359939,
			210719881,
			421439783,
			842879579,
			1685759167,

		//chunk #2
		433,
			877,
			1759,
			3527,
			7057,
			14143,
			28289,
			56591,
			113189,
			226379,
			452759,
			905551,
			1811107,
			3622219,
			7244441,
			14488931,
			28977863,
			57955739,
			115911563,
			231823147,
			463646329,
			927292699,
			1854585413,

		//chunk #3
		953,
			1907,
			3821,
			7643,
			15287,
			30577,
			61169,
			122347,
			244703,
			489407,
			978821,
			1957651,
			3915341,
			7830701,
			15661423,
			31322867,
			62645741,
			125291483,
			250582987,
			501165979,
			1002331963,
			2004663929,

		//chunk #4
		1039,
			2081,
			4177,
			8363,
			16729,
			33461,
			66923,
			133853,
			267713,
			535481,
			1070981,
			2141977,
			4283963,
			8567929,
			17135863,
			34271747,
			68543509,
			137087021,
			274174111,
			548348231,
			1096696463,

		//chunk #5
		31,
			67,
			137,
			277,
			557,
			1117,
			2237,
			4481,
			8963,
			17929,
			35863,
			71741,
			143483,
			286973,
			573953,
			1147921,
			2295859,
			4591721,
			9183457,
			18366923,
			36733847,
			73467739,
			146935499,
			293871013,
			587742049,
			1175484103,

		//chunk #6
		599,
			1201,
			2411,
			4831,
			9677,
			19373,
			38747,
			77509,
			155027,
			310081,
			620171,
			1240361,
			2480729,
			4961459,
			9922933,
			19845871,
			39691759,
			79383533,
			158767069,
			317534141,
			635068283,
			1270136683,

		//chunk #7
		311,
			631,
			1277,
			2557,
			5119,
			10243,
			20507,
			41017,
			82037,
			164089,
			328213,
			656429,
			1312867,
			2625761,
			5251529,
			10503061,
			21006137,
			42012281,
			84024581,
			168049163,
			336098327,
			672196673,
			1344393353,

		//chunk #8
		3,
			7,
			17,
			37,
			79,
			163,
			331,
			673,
			1361,
			2729,
			5471,
			10949,
			21911,
			43853,
			87719,
			175447,
			350899,
			701819,
			1403641,
			2807303,
			5614657,
			11229331,
			22458671,
			44917381,
			89834777,
			179669557,
			359339171,
			718678369,
			1437356741,

		//chunk #9
		43,
			89,
			179,
			359,
			719,
			1439,
			2879,
			5779,
			11579,
			23159,
			46327,
			92657,
			185323,
			370661,
			741337,
			1482707,
			2965421,
			5930887,
			11861791,
			23723597,
			47447201,
			94894427,
			189788857,
			379577741,
			759155483,
			1518310967,

		//chunk #10
		379,
			761,
			1523,
			3049,
			6101,
			12203,
			24407,
			48817,
			97649,
			195311,
			390647,
			781301,
			1562611,
			3125257,
			6250537,
			12501169,
			25002389,
			50004791,
			100009607,
			200019221,
			400038451,
			800076929,
			1600153859 };

	static {
		Arrays.sort(primeCapacities);
	}

	/**
	 * Values stored in the vector
	 */
	private double[] values;

	/**
	 * Their indices
	 */
	private int[] indices;

	/**
	 * Which entries that are actually in use
	 */
	private boolean[] used;

	/**
	 * Size of the hash
	 */
	private int m;

	/**
	 * Current prime index
	 */
	private int primeI;

	/**
	 * Constructor for HashVector
	 * 
	 * @param size
	 *            Size of the vector
	 * @param nz
	 *            Initial preallocation
	 */
	public HashVector(int size, int nz) {
		super(size);

		construct(nz);
	}

	/**
	 * Constructor for HashVector
	 * 
	 * @param size
	 *            Size of the vector
	 */
	public HashVector(int size) {
		this(size, 0);
	}

	private void construct(int nz) {
		// Find first prime larger than nz. Speed doesn't matter much here
		for (primeI = 0; primeI < primeCapacities.length; ++primeI)
			if (primeCapacities[primeI] >= nz)
				break;

		m = primeCapacities[primeI];

		used = new boolean[m];
		indices = new int[m];
		values = new double[m];
	}

	/**
	 * Constructor for HashVector
	 * 
	 * @param x
	 *            Vector to copy contents from
	 * @param deep
	 *            True for a deep copy. For shallow copies, <code>x</code>
	 *            must be a <code>HashVector</code>
	 */
	public HashVector(Vector x, boolean deep) {
		super(x);

		if (deep) {
			construct(x.cardinality());
			/*for (VectorEntry e : x)
				set(e.index(), e.get());*/
			VectorEntry e;
			Iterator iter = x.iterator();
			while(iter.hasNext()) {
				e = (VectorEntry) iter.next();
				set(e.index(), e.get());
			}
		} else {
			HashVector xh = (HashVector) x;
			used = xh.getUsed();
			indices = xh.getIndices();
			values = xh.getValues();

			m = used.length;
			for (primeI = 0; primeI < primeCapacities.length; ++primeI)
				if (primeCapacities[primeI] == m)
					break;
		}
	}

	/**
	 * Constructor for HashVector
	 * 
	 * @param x
	 *            Vector to copy contents from
	 */
	public HashVector(Vector x) {
		this(x, true);
	}

	/**
	 * Returns an array indicating which entries are in use, and which are
	 * free.
	 */
	public boolean[] getUsed() {
		return used;
	}

	/**
	 * Returns the indices. Only indices of occupied cells are valid
	 */
	public int[] getIndices() {
		return indices;
	}

	/**
	 * Returns the entries. Only entries of occupied cells are valid
	 */
	public double[] getValues() {
		return values;
	}

	public void add(int index, double value) {
		check(index);

		int i = insertionIndex(index);
		values[i] += value;
	}

	public double get(int index) {
		check(index);

		int i = search(index);

		if (i != -1)
			return values[i];
		else
			return 0;
	}

	public void set(int index, double value) {
		check(index);

		int i = insertionIndex(index);
		values[i] = value;
	}

	/**
	 * Returns an index into the hash table which corresponds to the given
	 * vector index. This function always returns a valid index
	 */
	private int insertionIndex(int k) {
		int i = 0;
		while ((i = index(k)) == -1)
			rehash();
		return i;
	}

	/**
	 * Expands the hash table. The new size will be a prime
	 */
	private void rehash() {
		// The next larger prime
		m = primeCapacities[++primeI];

		// Get the old data
		int[] oldIndices = indices;
		double[] oldValues = values;
		boolean[] wasUsed = used;

		// Allocate new data
		indices = new int[m];
		values = new double[m];
		used = new boolean[m];

		// Move the old data over to the new arrays
		for (int i = 0; i < oldIndices.length; ++i)
			if (wasUsed[i])
				values[index(oldIndices[i])] = oldValues[i];
	}

	/**
	 * Finds where in the hash table to put a key. On sucess, the cell is
	 * marked as used, and the corresponding index is returned. On failure, the
	 * sentinel value -1 is returned, in which case the hash table must be
	 * increased in size, and this function should be called again
	 */
	private int index(int k) {
		for (int i = 0, j = h(k, i); i < m; j = h(k, ++i)) {
			if (used[j] && indices[j] == k)
				return j;
			else if (!used[j]) {
				used[j] = true;
				indices[j] = k;
				return j;
			}
		}

		// Not found, must rehash to make space available
		return -1;
	}

	/**
	 * Searches the hash table for the index of the given key. On sucess the
	 * correct index is returned, and on failure the sentinel -1 is returned
	 */
	private int search(int k) {
		for (int i = 0, j = h(k, i); used[j] && i < m; j = h(k, ++i))
			if (used[j] && indices[j] == k)
				return j;

		// Not found
		return -1;
	}

	/**
	 * Primary hash function
	 */
	private int h1(int k) {
		return k % m;
	}

	/**
	 * Auxiliary hash function
	 */
	private int h2(int k) {
		return 1 + k % (m - 1);
	}

	/**
	 * Probing hash function
	 */
	private int h(int k, int i) {
		return (h1(k) + i * h2(k)) % m;
	}

	public double dot(Vector y) {
		if (!(y instanceof DenseVector))
			return super.dot(y);

		checkSize(y);

		double[] yd = ((DenseVector) y).getData();

		double ret = 0;
		for (int i = 0; i < m; ++i)
			if (used[i])
				ret += values[i] * yd[indices[i]];
		return ret;
	}

	//public HashVector copy() {
	public Vector copy() {
		return new HashVector(this);
	}

	//public HashVector zero() {
	public Vector zero() {
		for (int i = 0; i < m; ++i)
			values[i] = 0;
		return this;
	}

	//public HashVector scale(double alpha) {
	public Vector scale(double alpha) {
		// Quick return if possible
		if (alpha == 0)
			return zero();
		else if (alpha == 1)
			return this;

		for (int i = 0; i < m; ++i)
			if (used[i])
				values[i] *= alpha;

		return this;
	}

	//public Iterator < VectorEntry > iterator() {
	public Iterator iterator() {
		return new HashVectorIterator();
	}

	/**
	 * Iterator over a hash vector
	 */
	private class HashVectorIterator extends AbstractVectorIterator {

		/**
		 * Current sparse cursor index
		 */
		private int sparseCursor;

		/**
		 * Next sparse cursor index
		 */
		private int sparseCursorNext;

		/**
		 * Constructor for HashVectorIterator
		 */
		public HashVectorIterator() {
			entry = new HashVectorEntry();
			init();
		}

		public boolean hasNext() {
			return sparseCursor < used.length;
		}

		protected void cycle() {
			super.cycle();
			sparseCursor = sparseCursorNext;
		}

		protected void updateEntry() {
			((HashVectorEntry) entry).update(
				indices[sparseCursor],
				values[sparseCursor],
				sparseCursor);
		}

		protected boolean hasNextNext() {
			return sparseCursorNext < used.length;
		}

		protected void nextPosition() {
			do {
				sparseCursorNext++;
			} while (
				hasNextNext() && (!used[sparseCursorNext] || nextValue() == 0));
		}

		protected double nextValue() {
			return values[sparseCursorNext];
		}
	}

	/**
	 * Entry returned when iterating over a hash vector
	 */
	private class HashVectorEntry extends RefVectorEntry {

		private int cursor;

		public void update(int index, double value, int cursor) {
			super.update(index, value);
			this.cursor = cursor;
		}

		public void set(double value) {
			this.value = value;
			values[cursor] = value;
		}

	}

}
