/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt;

import mt.DenseVector;
import mt.Matrix;
import mt.Vector;

/**
 * Matrix stored using an array of <code>HashVector</code>, column-wise
 */
public class HashColMatrix extends AbstractColMatrix {

	/**
	 * Constructor for HashColMatrix
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of column
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 */
	/*public HashColMatrix(int numRows, int numColumns, int numThreads) {
		super(numRows, numColumns, new HashVector(numRows), numThreads);
	}*/

	/**
	 * Constructor for HashColMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 * @param deep
	 *            True for a deep copy, false for a reference copy. A reference
	 *            copy can only be made of a <code>HashColMatrix</code>
	 */
	/*public HashColMatrix(Matrix A, int numThreads, boolean deep) {
		super(A, new HashVector(A.numRows()), numThreads, deep);
	}*/

	/**
	 * Constructor for HashColMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. The copy will be deep
	 * @param numThreads
	 *            Number of threads to use in parallelization
	 */
	/*public HashColMatrix(Matrix A, int numThreads) {
		super(A, new HashVector(A.numRows()), numThreads);
	}*/

	/**
	 * Constructor for HashColMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param numRows
	 *            Number of rows
	 * @param numColumns
	 *            Number of column
	 */
	public HashColMatrix(int numRows, int numColumns) {
		super(numRows, numColumns, new HashVector(numRows));
	}

	/**
	 * Constructor for HashColMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param A
	 *            Matrix to copy contents from. The copy will be deep
	 */
	public HashColMatrix(Matrix A) {
		super(A, new HashVector(A.numRows()));
	}

	/**
	 * Constructor for HashColMatrix. Tries to use as many threads as the
	 * current machine has processors
	 * 
	 * @param A
	 *            Matrix to copy contents from
	 * @param deep
	 *            True for a deep copy, false for a reference copy. A reference
	 *            copy can only be made of a <code>HashColMatrix</code>
	 */
	public HashColMatrix(Matrix A, boolean deep) {
		super(A, new HashVector(A.numRows()), deep);
	}

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.multAdd(alpha, x, beta, y, z);

		checkMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		// z = beta/alpha * y
		z.set(beta / alpha, y);

		// z = A*x + z
		for (int i = 0; i < numColumns; ++i) {
			HashVector v = (HashVector) colD[i];
			int[] index = v.getIndices();
			double[] data = v.getValues();
			boolean[] used = v.getUsed();
			for (int j = 0; j < used.length; ++j)
				if (used[j])
					zd[index[j]] += data[j] * xd[i];
		}

		// z = alpha*z = alpha*A*x + beta*y
		return z.scale(alpha);
	}

	//public HashColMatrix copy() {
	public Matrix copy() {
		return new HashColMatrix(this);
	}

	//public HashVector getColumn(int i) {
	public Vector getColumn(int i) {
		return (HashVector) colD[i];
	}

	/**
	 * Sets the given column equal the passed vector
	 */
	public void setRow(int i, HashVector x) {
		if (x.size() != numRows)
			throw new IllegalArgumentException("New column must be of the same size as existing column");
		colD[i] = x;
	}

}
