/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.BandMatrix;
import mt.DenseMatrix;
import mt.Matrix;
import mt.Matrix.Norm;
import mt.MatrixSingularException;
import mt.UnitLowerTriangBandMatrix;
import mt.UpperTriangBandMatrix;
import mt.ll.BLASkernel.Transpose;
import mt.ll.Interface;

/**
 * Banded LU decomposition
 */
public class BandLU {

	/**
	 * Holds the LU factors
	 */
	private BandMatrix LU;

	/**
	 * Row pivotations
	 */
	private int[] ipiv;

	/**
	 * True if the matrix was singular
	 */
	private boolean singular;

	/**
	 * Number of bands in the matrix A
	 */
	private int kl, ku;

	/**
	 * Constructor for BandLU
	 * 
	 * @param A
	 *            Matrix to decompose
	 * @param inplace
	 *            True if the decomposition overwrites <code>A</code>, else
	 *            false. For in-place decompositions, an additional
	 *            number of superdiagonals equal the number of subdiagonals
	 *            must be present. These additional superdiagonals are not
	 *            considered part of the matrix to decompose, and are only
	 *            used for workspace
	 */
	public BandLU(BandMatrix A, boolean inplace) {
		kl = A.numSubDiagonals();
		ku = A.numSuperDiagonals();
		if (inplace) {
			ku -= kl;
			if (ku < 0)
				throw new IllegalArgumentException("kl + ku superdiagonals needed for inplace factorization");
			LU = A;
		} else
			LU = new BandMatrix(A, kl, ku + kl);

		ipiv = new int[Math.min(A.numRows(), A.numColumns())];

		int info =
			Interface.lapack().gbtrf(
				A.numRows(),
				A.numColumns(),
				kl,
				ku,
				LU.getData(),
				ipiv);

		if (info > 0)
			singular = true;
		else if (info < 0)
			throw new IllegalArgumentException();
	}

	/**
	 * Constructor for BandLU
	 * 
	 * @param A
	 *            Matrix to decompose. It will not be overwritten
	 */
	public BandLU(BandMatrix A) {
		this(A, false);
	}

	/**
	 * Returns the lower triangular factor
	 */
	public UnitLowerTriangBandMatrix getL() {
		return new UnitLowerTriangBandMatrix(LU, LU.numSubDiagonals(), false);
	}

	/**
	 * Returns the upper triangular factor
	 */
	public UpperTriangBandMatrix getU() {
		return new UpperTriangBandMatrix(LU, LU.numSuperDiagonals(), false);
	}

	/**
	 * Returns the decomposition matrix
	 */
	public BandMatrix getLU() {
		return LU;
	}

	/**
	 * Returns the row pivots
	 */
	public int[] getPivots() {
		return ipiv;
	}

	/**
	 * Checks for singularity
	 */
	public boolean isSingular() {
		return singular;
	}

	/**
	 * Computes the reciprocal condition number, using either the
	 * infinity norm of the 1 norm.
	 * 
	 * @param A
	 *            The matrix this is a decomposition of
	 * @param norm
	 *            Either <code>Norm.One</code> or <code>Norm.Infinity</code>
	 * @return    The reciprocal condition number. Values close to unity
	 *            indicate a well-conditioned system, while numbers close to
	 *            zero do not.
	 */
	public double rcond(Matrix A, Norm norm) {
		if (norm != Norm.One || norm != Norm.Infinity)
			throw new IllegalArgumentException("Only the 1 or the Infinity norms are supported");

		double anorm = A.norm(norm);

		int n = A.numRows();
		double[] work = new double[3 * n];
		int[] lwork = new int[n];

		double[] rcond = new double[1];
		int info =
			Interface.lapack().gbcon(
				norm,
				A.numRows(),
				kl,
				ku,
				LU.getData(),
				ipiv,
				anorm,
				rcond,
				work,
				lwork);

		if (info < 0)
			throw new IllegalArgumentException();

		return rcond[0];
	}

	/**
	 * Computes <code>A\B</code>, overwriting <code>B</code>
	 */
	public DenseMatrix solve(DenseMatrix B) {
		return solve(B, Transpose.NoTranspose);
	}

	/**
	 * Computes <code>A<sup>T</sup>\B</code>, overwriting <code>B</code>
	 */
	public DenseMatrix transSolve(DenseMatrix B) {
		return solve(B, Transpose.Transpose);
	}

	private DenseMatrix solve(DenseMatrix B, Transpose trans) {
		if (singular)
			throw new MatrixSingularException();

		int info =
			Interface.lapack().gbtrs(
				trans,
				LU.numRows(),
				kl,
				ku,
				B.numColumns(),
				LU.getData(),
				ipiv,
				B.getData());

		if (info < 0)
			throw new IllegalArgumentException();

		return B;
	}
}
