/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

//import java.util.Formatter;
import java.util.Iterator;

/**
 * Partial implementation of <code>Matrix</code>. Subclasses should provide
 * implementations of the following methods:
 * <ul>
 * <li>Basic operations:</li>
 * <ul>
 * <li>{@link #add(int, int, double) add}</li>
 * <li>{@link #get(int, int) get}</li>
 * <li>{@link #set(int, int, double) set}</li>
 * <li>{@link #copy() copy}</li>
 * <li>{@link #zero() zero}</li>
 * </ul>
 * <li>Kernel mathematical operations. Default implementations are provided,
 * but subclasses can provide faster version. Kernel operations are:</li>
 * <ul>
 * <li>{@link #multAdd(double, mt.Vector, double, mt.Vector, mt.Vector) multAdd}</li>
 * <li>
 * {@link #transMultAdd(double, mt.Vector, double, mt.Vector, mt.Vector) transMultAdd}
 * </li>
 * <li>{@link #multAdd(double, mt.Matrix, double, mt.Matrix, mt.Matrix) multAdd}</li>
 * <li>
 * {@link #transAmultAdd(double, mt.Matrix, double, mt.Matrix, mt.Matrix) transAmultAdd}
 * </li>
 * <li>
 * {@link #transBmultAdd(double, mt.Matrix, double, mt.Matrix, mt.Matrix) transBmultAdd}
 * </li>
 * <li>
 * {@link #transABmultAdd(double, mt.Matrix, double, mt.Matrix, mt.Matrix) transABmultAdd}
 * </li>
 * </ul>
 * <li>Faster iterators. Most other methods in the class uses an iterator, and
 * an efficient iterator can yield substantial improvements in performance. The
 * default iterator simply calls {@link #get(int, int) get}.</li>
 * <li>Solvers. These throw <code>UnsupportedOperationException</code>
 * by default, however the subclasses in MT all implement the direct solvers
 * correctly by calling LAPACK.</li>
 * </ul>
 */
public abstract class AbstractMatrix implements Matrix {

	/**
	 * Number of rows
	 */
	protected int numRows;

	/**
	 * Number of columns
	 */
	protected int numColumns;

	/**
	 * Constructor for AbstractMatrix.
	 */
	protected AbstractMatrix(int numRows, int numColumns) {
		if (numRows < 0 || numColumns < 0)
			throw new IllegalArgumentException("Matrix size cannot be negative");
		this.numRows = numRows;
		this.numColumns = numColumns;
	}

	/**
	 * Constructor for AbstractMatrix, same size as A.
	 */
	protected AbstractMatrix(Matrix A) {
		this(A.numRows(), A.numColumns());
	}

	public int numRows() {
		return numRows;
	}

	public int numColumns() {
		return numColumns;
	}

	public boolean isSquare() {
		return numRows == numColumns;
	}

	/**
	 * Checks the passes row and column indices
	 */
	protected void check(int row, int column) {
		if (row < 0)
			throw new IllegalArgumentException("row index is negative");
		if (column < 0)
			throw new IllegalArgumentException("column index is negative");
		if (row >= numRows)
			throw new IllegalArgumentException("row index >= numRows");
		if (column >= numColumns)
			throw new IllegalArgumentException("column index >= numColumns");
	}

	/**
	 * Checks the row and column indices against the matrix and each other
	 */
	protected void check(int[] row, int[] column) {
		for (int i = 0; i < row.length; ++i)
			for (int j = 0; j < column.length; ++j)
				check(row[i], column[j]);
	}

	/**
	 * Checks the indices and the value array for consistency issues
	 */
	protected void check(int[] row, int[] column, double[][] values) {
		if (row.length != values.length)
			throw new IllegalArgumentException("row.length != values.length");

		for (int i = 0; i < row.length; ++i) {
			if (column.length != values[i].length)
				throw new IllegalArgumentException("column.length != values[i].length");
			for (int j = 0; j < column.length; ++j)
				check(row[i], column[j]);
		}
	}

	public void add(int[] row, int[] column, double[][] values) {
		check(row, column, values);

		for (int j = 0; j < column.length; ++j)
			for (int i = 0; i < row.length; ++i)
				add(row[i], column[j], values[i][j]);
	}

	public double[][] get(int[] row, int[] column, double[][] values) {
		check(row, column, values);

		for (int j = 0; j < column.length; ++j)
			for (int i = 0; i < row.length; ++i)
				values[i][j] = get(row[i], column[j]);
		return values;
	}

	public void set(int[] row, int[] column, double[][] values) {
		check(row, column, values);

		for (int j = 0; j < column.length; ++j)
			for (int i = 0; i < row.length; ++i)
				set(row[i], column[j], values[i][j]);
	}

	public void add(int row, int column, double value) {
		set(row, column, value + get(row, column));
	}

	public double get(int row, int column) {
		throw new UnsupportedOperationException();
	}

	public void set(int row, int column, double value) {
		throw new UnsupportedOperationException();
	}

	public Matrix copy() {
		throw new UnsupportedOperationException();
	}

	public Matrix zero() {
		/*for (MatrixEntry e : this)
			e.set(0);*/
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			e.set(0);
		}
		return this;
	}

	public String toString() {
		// Output into coordinate format. Indices start from 1 instead of 0
		//Formatter out = new Formatter();
		//out.format("%10d %10d %19d\n", numRows, numColumns, cardinality());
		String s = Integer.toString(numRows) + " " + numColumns + " " + cardinality() + "\n";
		//for (MatrixEntry e : this)
			/*out.format(
				"%10d %10d % .12e\n",
				e.row() + 1,
				e.column() + 1,
				e.get()); */
			//s += (e.row() + 1) + " " + (e.column() + 1) + " " + e.get() + "\n";
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			s += (e.row() + 1) + " " + (e.column() + 1) + " " + e.get() + "\n";
		}
		
		return s;
	}

	public Matrix mult(double alpha, Matrix B, Matrix C) {
		return multAdd(alpha, B, 0, C, C);
	}

	public Vector mult(double alpha, Vector x, Vector y) {
		return multAdd(alpha, x, 0, y, y);
	}

	public Matrix mult(Matrix B, Matrix C) {
		return multAdd(1, B, 0, C, C);
	}

	public Vector mult(Vector x, Vector y) {
		return multAdd(1, x, 0, y, y);
	}

	public Matrix multAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		checkMultAdd(B, C, D);

		D.set(beta, C);

		if (alpha != 0.)
			for (int i = 0; i < numRows; ++i)
				for (int j = 0; j < C.numColumns(); ++j) {
					double dot = 0;
					for (int k = 0; k < numColumns; ++k)
						dot += get(i, k) * B.get(k, j);
					D.add(i, j, alpha * dot);
				}

		return D;
	}

	/**
	 * Checks the arguments to <code>transAmultAdd</code>
	 */
	protected void checkMultAdd(Matrix B, Matrix C, Matrix D) {
		if (numRows != C.numRows())
			throw new IllegalArgumentException("A.numRows() != C.numRows()");
		if (numRows != D.numRows())
			throw new IllegalArgumentException("A.numRows() != D.numRows()");
		if (numColumns != B.numRows())
			throw new IllegalArgumentException("A.numColumns() != B.numRows()");
		if (B.numColumns() != C.numColumns())
			throw new IllegalArgumentException("B.numColumns() != C.numColumns()");
		if (C.numColumns() != D.numColumns())
			throw new IllegalArgumentException("C.numColulmns() != D.numColumns()");
	}

	public Matrix multAdd(Matrix B, double beta, Matrix C, Matrix D) {
		return multAdd(1, B, beta, C, D);
	}

	public Matrix multAdd(double alpha, Matrix B, Matrix C, Matrix D) {
		return multAdd(alpha, B, 1, C, D);
	}

	public Matrix multAdd(Matrix B, Matrix C, Matrix D) {
		return multAdd(1, B, 1, C, D);
	}

	public Matrix multAdd(double alpha, Matrix B, double beta, Matrix C) {
		return multAdd(alpha, B, beta, C, C);
	}

	public Matrix multAdd(double alpha, Matrix B, Matrix C) {
		return multAdd(alpha, B, 1, C, C);
	}

	public Matrix multAdd(Matrix B, double beta, Matrix C) {
		return multAdd(1, B, beta, C, C);
	}

	public Matrix multAdd(Matrix B, Matrix C) {
		return multAdd(1, B, 1, C, C);
	}

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		checkMultAdd(x, y, z);

		z.set(beta, y);

		if (alpha != 0)
			for (int i = 0; i < numRows; ++i) {
				double dot = 0;
				for (int j = 0; j < numColumns; ++j)
					dot += get(i, j) * x.get(j);
				z.add(i, alpha * dot);
			}

		return z;
	}

	/**
	 * Checks the arguments to <code>multAdd</code>
	 */
	protected void checkMultAdd(Vector x, Vector y, Vector z) {
		if (numColumns != x.size())
			throw new IllegalArgumentException("A.numColumns() != x.size()");
		if (numRows != y.size())
			throw new IllegalArgumentException("A.numRows() != y.size()");
		if (numRows != z.size())
			throw new IllegalArgumentException("A.numRows() != z.size()");
	}

	public Vector multAdd(double alpha, Vector x, double beta, Vector y) {
		return multAdd(alpha, x, beta, y, y);
	}

	public Vector multAdd(double alpha, Vector x, Vector y, Vector z) {
		return multAdd(alpha, x, 1, y, z);
	}

	public Vector multAdd(double alpha, Vector x, Vector y) {
		return multAdd(alpha, x, 1, y, y);
	}

	public Vector multAdd(Vector x, double beta, Vector y, Vector z) {
		return multAdd(1, x, beta, y, z);
	}

	public Vector multAdd(Vector x, double beta, Vector y) {
		return multAdd(1, x, beta, y, y);
	}

	public Vector multAdd(Vector x, Vector y, Vector z) {
		return multAdd(1, x, 1, y, z);
	}

	public Vector multAdd(Vector x, Vector y) {
		return multAdd(1, x, 1, y, y);
	}

	public Matrix rank1(double alpha, Matrix C, double beta) {
		checkRank1(C);

		if (alpha == 0)
			return scale(beta);

		return C.transBmultAdd(alpha, C, beta, this);
	}

	/**
	 * Checks that a matrix rank1 update is possible for the given matrix
	 */
	protected void checkRank1(Matrix C) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (numRows != C.numRows())
			throw new IllegalArgumentException("A.numRows() != C.numRows()");
	}

	public Matrix rank1(double alpha, Matrix C) {
		return rank1(alpha, C, 1);
	}

	public Matrix rank1(double alpha, Vector x, Vector y) {
		checkRank1(x, y);

		if (alpha == 0)
			return this;

		VectorEntry ei, ej;
		Iterator iterX = x.iterator();
		while(iterX.hasNext()) {
			ei = (VectorEntry) iterX.next();
			Iterator iterY = y.iterator();
			while(iterY.hasNext()) {
				ej = (VectorEntry) iterY.next();
				add(ei.index(), ej.index(), alpha * ei.get() * ej.get());
			}
		}
		
		
		/*for (VectorEntry ei : x)
			for (VectorEntry ej : y)
				add(ei.index(), ej.index(), alpha * ei.get() * ej.get());*/

		return this;
	}

	/**
	 * Checks that a vector rank1 update is possible for the given
	 * vectors
	 */
	protected void checkRank1(Vector x, Vector y) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (x.size() != numRows)
			throw new IllegalArgumentException("x.size() != A.numRows()");
		if (y.size() != numColumns)
			throw new IllegalArgumentException("y.size() != A.numColumns()");
	}

	public Matrix rank1(Matrix C, double beta) {
		return rank1(1, C, beta);
	}

	public Matrix rank1(Matrix C) {
		return rank1(1, C, 1);
	}

	public Matrix rank1(Vector x, Vector y) {
		return rank1(1, x, y);
	}

	public Matrix rank1(Vector x) {
		return rank1(1, x, x);
	}

	public Matrix rank2(double alpha, Matrix B, Matrix C, double beta) {
		checkRank2(B, C);

		if (alpha == 0)
			return scale(beta);

		return B.transBmultAdd(alpha, C, C.transBmultAdd(alpha, B, beta, this));
	}

	/**
	 * Checks that a rank2 update is legal for the given arguments
	 */
	protected void checkRank2(Matrix B, Matrix C) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (B.numRows() != C.numRows())
			throw new IllegalArgumentException("B.numRows() != C.numRows()");
		if (B.numColumns() != C.numColumns())
			throw new IllegalArgumentException("B.numColumns() != C.numColumns()");
	}

	public Matrix rank2(double alpha, Matrix B, Matrix C) {
		return rank2(alpha, B, C, 1);
	}

	public Matrix rank2(double alpha, Vector x, Vector y) {
		checkRank2(x, y);

		if (alpha == 0)
			return this;

		VectorEntry ei, ej;
		Iterator iterX = x.iterator();
		while(iterX.hasNext()) {
			ei = (VectorEntry) iterX.next();
			Iterator iterY = y.iterator();
			while(iterY.hasNext()) {
				ej = (VectorEntry) iterY.next();
				add(ei.index(), ej.index(), alpha * ei.get() * ej.get());
				add(ej.index(), ei.index(), alpha * ei.get() * ej.get());
			}
		}
		
		/*for (VectorEntry ei : x)
			for (VectorEntry ej : y) {
				add(ei.index(), ej.index(), alpha * ei.get() * ej.get());
				add(ej.index(), ei.index(), alpha * ei.get() * ej.get());
			}*/

		return this;
	}

	/**
	 * Checks that a vector rank2 update is legal with the given vectors
	 */
	protected void checkRank2(Vector x, Vector y) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (x.size() != numRows)
			throw new IllegalArgumentException("x.size() != A.numRows()");
		if (y.size() != numRows)
			throw new IllegalArgumentException("y.size() != A.numRows()");
	}

	public Matrix rank2(Matrix B, Matrix C, double beta) {
		return rank2(1, B, C, beta);
	}

	public Matrix rank2(Matrix B, Matrix C) {
		return rank2(1, B, C, 1);
	}

	public Matrix rank2(Vector x, Vector y) {
		return rank2(1, x, y);
	}

	public Matrix scale(double alpha) {
		if (alpha == 1)
			return this;
		else if (alpha == 0)
			return zero();

		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			e.set(alpha * e.get());
		}
		
		/*for (MatrixEntry e : this)
			e.set(alpha * e.get());*/

		return this;
	}

	public Vector solve(Vector b, Vector x) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Checks that a matrix inversion is legal for the given arguments.
	 * This is for the square case, not for least-squares problems
	 */
	protected void checkSolve(Vector b, Vector x) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (b.size() != numRows)
			throw new IllegalArgumentException("b.size() != A.numRows()");
		if (x.size() != numColumns)
			throw new IllegalArgumentException("x.size() != A.numColumns()");
	}

	public Matrix transABmult(double alpha, Matrix B, Matrix C) {
		return transABmultAdd(alpha, B, 0, C, C);
	}

	public Matrix transABmult(Matrix B, Matrix C) {
		return transABmultAdd(1, B, 0, C, C);
	}

	public Matrix transABmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		checkTransABmultAdd(B, C, D);

		D.set(beta, C);

		if (alpha != 0)
			for (int i = 0; i < numColumns; ++i)
				for (int j = 0; j < C.numColumns(); ++j) {
					double dot = 0;
					for (int k = 0; k < numRows; ++k)
						dot += get(k, i) * B.get(j, k);
					D.add(i, j, alpha * dot);
				}

		return D;
	}

	/**
	 * Checks the arguments to <code>transABmultAdd</code>
	 */
	protected void checkTransABmultAdd(Matrix B, Matrix C, Matrix D) {
		if (numRows != B.numColumns())
			throw new IllegalArgumentException("A.numRows() != B.numColumns()");
		if (numColumns != C.numRows())
			throw new IllegalArgumentException("A.numColumns() != C.numRows()");
		if (numColumns != D.numRows())
			throw new IllegalArgumentException("A.numColumns() != D.numRows()");
		if (B.numRows() != C.numColumns())
			throw new IllegalArgumentException("B.numRows() != C.numColumns()");
		if (B.numRows() != D.numColumns())
			throw new IllegalArgumentException("B.numRows() != D.numColumns()");
	}

	public Matrix transABmultAdd(Matrix B, double beta, Matrix C, Matrix D) {
		return transABmultAdd(1, B, beta, C, D);
	}

	public Matrix transABmultAdd(double alpha, Matrix B, Matrix C, Matrix D) {
		return transABmultAdd(alpha, B, 1, C, D);
	}

	public Matrix transABmultAdd(Matrix B, Matrix C, Matrix D) {
		return transABmultAdd(1, B, 1, C, D);
	}

	public Matrix transABmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C) {
		return transABmultAdd(alpha, B, beta, C, C);
	}

	public Matrix transABmultAdd(double alpha, Matrix B, Matrix C) {
		return transABmultAdd(alpha, B, 1, C, C);
	}

	public Matrix transABmultAdd(Matrix B, double beta, Matrix C) {
		return transABmultAdd(1, B, beta, C, C);
	}

	public Matrix transABmultAdd(Matrix B, Matrix C) {
		return transABmultAdd(1, B, 1, C, C);
	}

	public Matrix transAmult(double alpha, Matrix B, Matrix C) {
		return transAmultAdd(alpha, B, 0, C, C);
	}

	public Matrix transAmult(Matrix B, Matrix C) {
		return transAmultAdd(1, B, 0, C, C);
	}

	public Matrix transAmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		checkTransAmultAdd(B, C, D);

		D.set(beta, C);

		if (alpha != 0)
			for (int i = 0; i < numColumns; ++i)
				for (int j = 0; j < C.numColumns(); ++j) {
					double dot = 0;
					for (int k = 0; k < numRows; ++k)
						dot += get(k, i) * B.get(k, j);
					D.add(i, j, alpha * dot);
				}

		return D;
	}

	/**
	 * Checks the arguments to <code>transAmultAdd</code>
	 */
	protected void checkTransAmultAdd(Matrix B, Matrix C, Matrix D) {
		if (numRows != B.numRows())
			throw new IllegalArgumentException("A.numRows() != B.numRows()");
		if (numColumns != C.numRows())
			throw new IllegalArgumentException("A.numColumns() != C.numRows()");
		if (numColumns != D.numRows())
			throw new IllegalArgumentException("A.numColumns() != D.numRows()");
		if (B.numColumns() != C.numColumns())
			throw new IllegalArgumentException("B.numColumns() != C.numColumns()");
		if (B.numColumns() != D.numColumns())
			throw new IllegalArgumentException("B.numColumns() != D.numColumns()");
	}

	public Matrix transAmultAdd(Matrix B, double beta, Matrix C, Matrix D) {
		return transAmultAdd(1, B, beta, C, D);
	}

	public Matrix transAmultAdd(double alpha, Matrix B, Matrix C, Matrix D) {
		return transAmultAdd(alpha, B, 1, C, D);
	}

	public Matrix transAmultAdd(Matrix B, Matrix C, Matrix D) {
		return transAmultAdd(1, B, 1, C, D);
	}

	public Matrix transAmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C) {
		return transAmultAdd(alpha, B, beta, C, C);
	}

	public Matrix transAmultAdd(double alpha, Matrix B, Matrix C) {
		return transAmultAdd(alpha, B, 1, C, C);
	}

	public Matrix transAmultAdd(Matrix B, double beta, Matrix C) {
		return transAmultAdd(1, B, beta, C, C);
	}

	public Matrix transAmultAdd(Matrix B, Matrix C) {
		return transAmultAdd(1, B, 1, C, C);
	}

	public Matrix transBmult(double alpha, Matrix B, Matrix C) {
		return transBmultAdd(alpha, B, 0, C, C);
	}

	public Matrix transBmult(Matrix B, Matrix C) {
		return transBmultAdd(1, B, 0, C, C);
	}

	public Matrix transBmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C,
		Matrix D) {
		checkTransBmultAdd(B, C, D);

		D.set(beta, C);

		if (alpha != 0)
			for (int i = 0; i < numRows; ++i)
				for (int j = 0; j < C.numColumns(); ++j) {
					double dot = 0;
					for (int k = 0; k < numColumns; ++k)
						dot += get(i, k) * B.get(j, k);
					D.add(i, j, alpha * dot);
				}

		return D;
	}

	/**
	 * Checks the arguments to <code>transBmultAdd</code>
	 */
	protected void checkTransBmultAdd(Matrix B, Matrix C, Matrix D) {
		if (numColumns != B.numColumns())
			throw new IllegalArgumentException("A.numColumns() != B.numColumns()");
		if (numRows != C.numRows())
			throw new IllegalArgumentException("A.numRows() != C.numRows()");
		if (numRows != D.numRows())
			throw new IllegalArgumentException("A.numRows() != D.numRows()");
		if (B.numRows() != C.numColumns())
			throw new IllegalArgumentException("B.numRows() != C.numColumns()");
		if (B.numRows() != D.numColumns())
			throw new IllegalArgumentException("B.numRows() != D.numColumns()");
	}

	public Matrix transBmultAdd(Matrix B, double beta, Matrix C, Matrix D) {
		return transBmultAdd(1, B, beta, C, D);
	}

	public Matrix transBmultAdd(double alpha, Matrix B, Matrix C, Matrix D) {
		return transBmultAdd(alpha, B, 1, C, D);
	}

	public Matrix transBmultAdd(Matrix B, Matrix C, Matrix D) {
		return transBmultAdd(1, B, 1, C, D);
	}

	public Matrix transBmultAdd(
		double alpha,
		Matrix B,
		double beta,
		Matrix C) {
		return transBmultAdd(alpha, B, beta, C, C);
	}

	public Matrix transBmultAdd(double alpha, Matrix B, Matrix C) {
		return transBmultAdd(alpha, B, 1, C, C);
	}

	public Matrix transBmultAdd(Matrix B, double beta, Matrix C) {
		return transBmultAdd(1, B, beta, C, C);
	}

	public Matrix transBmultAdd(Matrix B, Matrix C) {
		return transBmultAdd(1, B, 1, C, C);
	}

	public Vector transMult(double alpha, Vector x, Vector y) {
		return transMultAdd(alpha, x, 0, y, y);
	}

	public Vector transMult(Vector x, Vector y) {
		return transMultAdd(1, x, 0, y, y);
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		checkTransMultAdd(x, y, z);

		z.set(beta, y);

		if (alpha != 0)
			for (int i = 0; i < numColumns; ++i) {
				double dot = 0;
				for (int j = 0; j < numRows; ++j)
					dot += get(j, i) * x.get(j);
				z.add(i, alpha * dot);
			}

		return z;
	}

	/**
	 * Checks the arguments to <code>transMultAdd</code>
	 */
	protected void checkTransMultAdd(Vector x, Vector y, Vector z) {
		if (numRows != x.size())
			throw new IllegalArgumentException("A.numRows() != x.size()");
		if (numColumns != y.size())
			throw new IllegalArgumentException("A.numColumns() != y.size()");
		if (numColumns != z.size())
			throw new IllegalArgumentException("A.numColumns() != z.size()");
	}

	public Vector transMultAdd(double alpha, Vector x, double beta, Vector y) {
		return transMultAdd(alpha, x, beta, y, y);
	}

	public Vector transMultAdd(double alpha, Vector x, Vector y, Vector z) {
		return transMultAdd(alpha, x, 1, y, z);
	}

	public Vector transMultAdd(double alpha, Vector x, Vector y) {
		return transMultAdd(alpha, x, 1, y, y);
	}

	public Vector transMultAdd(Vector x, double beta, Vector y, Vector z) {
		return transMultAdd(1, x, beta, y, z);
	}

	public Vector transMultAdd(Vector x, double beta, Vector y) {
		return transMultAdd(1, x, beta, y, y);
	}

	public Vector transMultAdd(Vector x, Vector y, Vector z) {
		return transMultAdd(1, x, 1, y, z);
	}

	public Vector transMultAdd(Vector x, Vector y) {
		return transMultAdd(1, x, 1, y, y);
	}

	public Matrix transRank1(double alpha, Matrix C, double beta) {
		checkTransRank1(C);

		if (alpha == 0)
			return scale(beta);

		return C.transAmultAdd(alpha, C, beta, this);
	}

	/**
	 * Checks that a transposed rank1 update is leagal with the given
	 * argument
	 */
	protected void checkTransRank1(Matrix C) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (numRows != C.numColumns())
			throw new IllegalArgumentException("A.numRows() != C.numColumns()");
	}

	public Matrix transRank1(double alpha, Matrix C) {
		return transRank1(alpha, C, 1);
	}

	public Matrix transRank1(Matrix C, double beta) {
		return transRank1(1, C, beta);
	}

	public Matrix transRank1(Matrix C) {
		return transRank1(1, C, 1);
	}

	public Matrix transRank2(double alpha, Matrix B, Matrix C, double beta) {
		checkTransRank2(B, C);

		if (alpha == 0)
			return scale(beta);

		return B.transAmultAdd(alpha, C, C.transAmultAdd(alpha, B, beta, this));
	}

	/**
	 * Checks that a transposed rank2 update is leagal with the given
	 * arguments
	 */
	protected void checkTransRank2(Matrix B, Matrix C) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (numRows != B.numColumns())
			throw new IllegalArgumentException("A.numRows() != B.numColumns()");
		if (B.numRows() != C.numRows())
			throw new IllegalArgumentException("B.numRows() != C.numRows()");
		if (B.numColumns() != C.numColumns())
			throw new IllegalArgumentException("B.numColumns() != C.numColumns()");
	}

	public Matrix transRank2(double alpha, Matrix B, Matrix C) {
		return transRank2(alpha, B, C, 1);
	}

	public Matrix transRank2(Matrix B, Matrix C, double beta) {
		return transRank2(1, B, C, beta);
	}

	public Matrix transRank2(Matrix B, Matrix C) {
		return transRank2(1, B, C, 1);
	}

	public Vector transSolve(Vector b, Vector x) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Checks that a transpose matrix inversion is legal for the given
	 * arguments. This is for the square case, not for least-squares
	 * problems
	 */
	protected void checkTransSolve(Vector b, Vector x) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (b.size() != numColumns)
			throw new IllegalArgumentException("b.size() != A.numColumns()");
		if (x.size() != numRows)
			throw new IllegalArgumentException("x.size() != A.numRows()");
	}

	public double norm(Norm type) {
		if (type == Norm.One)
			return norm1();
		else if (type == Norm.Frobenius)
			return normF();
		else if (type == Norm.Infinity)
			return normInf();
		else // Maxvalue
			return max();
	}

	/**
	 * Computes the 1 norm
	 */
	protected double norm1() {
		double[] rowSum = new double[numRows];
		//for (MatrixEntry e : this)
		//	rowSum[e.row()] += Math.abs(e.get());
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			rowSum[e.row()] += Math.abs(e.get());
		}
		return max(rowSum);
	}

	/**
	 * Computes the Frobenius norm. This implementation is overflow resistant
	 */
	protected double normF() {
		double scale = 0, ssq = 1;
		/*for (MatrixEntry e : this) {
			double Aval = e.get();
			if (Aval != 0) {
				double absxi = Math.abs(Aval);
				if (scale < absxi) {
					ssq = 1 + ssq * Math.pow(scale / absxi, 2);
					scale = absxi;
				} else
					ssq = ssq + Math.pow(absxi / scale, 2);
			}
		}*/
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			double Aval = e.get();
			if (Aval != 0) {
				double absxi = Math.abs(Aval);
				if (scale < absxi) {
					ssq = 1 + ssq * Math.pow(scale / absxi, 2);
					scale = absxi;
				} else
					ssq = ssq + Math.pow(absxi / scale, 2);
			}
		}
		return scale * Math.sqrt(ssq);
	}

	/**
	 * Computes the infinity norm
	 */
	protected double normInf() {
		double[] columnSum = new double[numColumns];
		/*for (MatrixEntry e : this)
			columnSum[e.column()] += Math.abs(e.get());*/
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			columnSum[e.column()] += Math.abs(e.get());
		}
		return max(columnSum);
	}

	/**
	 * Returns the largest absolute value
	 */
	protected double max() {
		double max = 0;
		/*for (MatrixEntry e : this)
			max = Math.max(Math.abs(e.get()), max);*/
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			max = Math.max(Math.abs(e.get()), max);
		}
		
		return max;
	}

	/**
	 * Returns the largest element of the passed array
	 */
	protected double max(double[] x) {
		double max = 0;
		for (int i = 0; i < x.length; ++i)
			max = Math.max(x[i], max);
		return max;
	}

	public Matrix transpose() {
		checkTranspose();

		for (int j = 0; j < numColumns; ++j)
			for (int i = j; i < numRows; ++i) {
				double value = get(i, j);
				set(i, j, get(j, i));
				set(j, i, value);
			}

		return this;
	}

	/**
	 * Checks that the matrix may be tranposed
	 */
	protected void checkTranspose() {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
	}

	public Matrix transpose(Matrix B) {
		checkTranspose(B);

		if (B == this)
			return transpose();

		B.zero();
		/*for (MatrixEntry e : this)
			B.set(e.column(), e.row(), e.get());*/
		MatrixEntry e;
		Iterator iter = iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			B.set(e.column(), e.row(), e.get());
		}

		return B;
	}

	/**
	 * Checks that this matrix can be transposed into the given matrix
	 */
	protected void checkTranspose(Matrix B) {
		if (numRows != B.numColumns())
			throw new IllegalArgumentException("A.numRows() != B.numColumns()");
		if (numColumns != B.numRows())
			throw new IllegalArgumentException("A.numColumns() != B.numRows()");
	}

	public Matrix add(double alpha, Matrix B) {
		checkSize(B);

		MatrixEntry e;
		Iterator iter;
		
		if (alpha != 0) {
			iter = B.iterator();
			while(iter.hasNext()) {
				e = (MatrixEntry) iter.next();
				add(e.row(), e.column(), alpha * e.get());
			}
		}
			/*for (MatrixEntry e : B)
				add(e.row(), e.column(), alpha * e.get());*/
			

		return this;
	}

	public Matrix add(Matrix B) {
		return add(1, B);
	}

	public Matrix set(double alpha, Matrix B) {
		checkSize(B);

		if (alpha == 0.)
			return zero();
		if (B == this)
			return scale(alpha);

		zero();
		/*for (MatrixEntry e : B)
			set(e.row(), e.column(), alpha * e.get());*/
		MatrixEntry e;
		Iterator iter = B.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			set(e.row(), e.column(), alpha * e.get());
		}

		return this;
	}

	/**
	 * Checks that the sizes of this matrix and the given conform
	 */
	protected void checkSize(Matrix B) {
		if (numRows != B.numRows())
			throw new IllegalArgumentException("A.numRows() != B.numRows()");
		if (numColumns != B.numColumns())
			throw new IllegalArgumentException("A.numColumns() != B.numColumns()");
	}

	public Matrix set(Matrix B) {
		return set(1, B);
	}

	public Matrix solve(Matrix B, Matrix X) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Checks that a matrix inversion is legal for the given arguments.
	 * This is for the square case, not for least-squares problems
	 */
	protected void checkSolve(Matrix B, Matrix X) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (B.numRows() != numRows)
			throw new IllegalArgumentException("B.numRows() != A.numRows()");
		if (B.numColumns() != X.numColumns())
			throw new IllegalArgumentException("B.numColumns() != X.numColumns()");
		if (X.numRows() != numColumns)
			throw new IllegalArgumentException("X.numRows() != A.numColumns()");
	}

	public Matrix transSolve(Matrix B, Matrix X) {
		throw new UnsupportedOperationException();
	}

	/**
	 * Checks that a transpose matrix inversion is legal for the given
	 * arguments. This is for the square case, not for least-squares
	 * problems
	 */
	protected void checkTransSolve(Matrix B, Matrix X) {
		if (!isSquare())
			throw new IllegalArgumentException("!A.isSquare()");
		if (B.numRows() != numColumns)
			throw new IllegalArgumentException("B.numRows() != A.numColumns()");
		if (B.numColumns() != X.numColumns())
			throw new IllegalArgumentException("B.numColumns() != X.numColumns()");
		if (X.numRows() != numRows)
			throw new IllegalArgumentException("X.numRows() != A.numRows()");
	}

	public int cardinality() {
		int nz = 0;
		/*for (MatrixEntry e : this)
			nz++;*/
		for(Iterator iter = iterator(); iter.hasNext(); iter.next(), ++nz) {}
		return nz;
	}

	public Matrix addDiagonal(double shift) {
		int max = Math.min(numRows, numColumns);
		for (int i = 0; i < max; ++i)
			add(i, i, shift);
		return this;
	}

	//public Iterator < MatrixEntry > iterator() {
	public Iterator iterator() {
		return new RefMatrixIterator();
	}

	/**
	 * Partial implementation of a matrix iterator. Subclasses must implement
	 * all abstract methods, and call <code>init</code> as the last step of
	 * the constructor. The entry must also be set.
	 */
	protected abstract class AbstractMatrixIterator
		implements Iterator {

		/**
		 * Current cursor indices
		 */
		protected int row, column;

		/**
		 * Next cursor indices
		 */
		protected int rowNext, columnNext;

		/**
		 * Recycled matrix entry
		 */
		protected MatrixEntry entry;

		/**
		 * Initializes the iterator. Positions the cursors, and sets the
		 * initial matrix entry
		 */
		protected void init() {
			// Position at next non-zero entry if necessary
			if (hasNextNext() && nextValue() == 0)
				rePositionNext();

			// Use that as current entry
			cycle();

			// Then find the next one. Needed when the user calls next()
			if (hasNextNext())
				rePositionNext();
		}

		public void remove() {
			entry.set(0);
		}

		//public MatrixEntry next() {
		public Object next() {
			// Sets the entry to the current position
			updateEntry();

			// Sets position to the next
			cycle();

			// Finds the next position if possible
			if (hasNextNext())
				rePositionNext();

			return entry;
		}

		public boolean hasNext() {
			return (row < numRows) && (column < numColumns);
		}

		/**
		 * Cycle the next position into the current
		 */
		protected void cycle() {
			row = rowNext;
			column = columnNext;
		}

		/**
		 * Sets the entry to the current position
		 */
		protected abstract void updateEntry();

		/**
		 * Moves the next-pointers to the next non-zero position
		 */
		protected void rePositionNext() {
			do {
				nextPosition();
			} while (hasNextNext() && nextValue() == 0);
		}

		/**
		 * Returns true if the next position is within bounds
		 */
		protected boolean hasNextNext() {
			return (rowNext < numRows) && (columnNext < numColumns);
		}

		/**
		 * Moves to the next entry, irrespective of wheter it's non-zero. It
		 * may move outside the bounds of the matrix only if there are no more
		 * entries left
		 */
		protected abstract void nextPosition();

		/**
		 * Returns value at the next position
		 */
		protected double nextValue() {
			return get(rowNext, columnNext);
		}

	}

	/**
	 * Iterator over a general matrix
	 */
	protected class RefMatrixIterator extends AbstractMatrixIterator {

		/**
		 * Constructor for RefMatrixIterator
		 */
		public RefMatrixIterator() {
			entry = new RefMatrixEntry();
			init();
		}

		protected void updateEntry() {
			((RefMatrixEntry) entry).update(row, column, get(row, column));
		}

		protected void nextPosition() {
			if (rowNext < numRows - 1)
				rowNext++;
			else {
				columnNext++;
				rowNext = 0;
			}
		}

	}

	/**
	 * Matrix entry backed by the matrix. May be recycled for higher performance
	 */
	protected class RefMatrixEntry implements MatrixEntry {

		/**
		 * Current matrix position
		 */
		protected int row, column;

		/**
		 * Value at current position
		 */
		protected double value;

		/**
		 * Updates the entry
		 */
		public void update(int row, int column, double value) {
			this.row = row;
			this.column = column;
			this.value = value;
		}

		public int row() {
			return row;
		}

		public int column() {
			return column;
		}

		public double get() {
			return value;
		}

		public void set(double value) {
			this.value = value;
			AbstractMatrix.this.set(row, column, value);
		}
	}

}
