/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.mixed;

import mt.Matrix;
import mt.Vector;
import smt.iter.IterativeSolverNotConvergedException;

/**
 * Uzawa algorithm using gradient approach.
 */
public class GradientUzawa extends AbstractMixedSolver {

	/**
	 * Temporary work vectors
	 */
	private Vector qr, p, h, r, t;

	/**
	 * Constructor for GradientUzawa
	 * 
	 * @param qTemplate
	 *            Vector of size <i>m</i>, used as template in preallocating
	 *            work vectors
	 * @param uTemplate
	 *            Vector of size <i>n</i>, used as template in preallocating
	 *            work vectors
	 */
	public GradientUzawa(Vector qTemplate, Vector uTemplate) {
		super(qTemplate);

		qr = qTemplate.copy();
		p = qTemplate.copy();
		h = qTemplate.copy();

		r = uTemplate.copy();
		t = uTemplate.copy();
	}

	protected void solveI(
		Matrix A,
		Matrix B,
		Matrix Bt,
		Matrix C,
		Vector q,
		Vector u,
		Vector f,
		Vector g)
		throws IterativeSolverNotConvergedException {

		C.multAdd(-1., u, g, t); // t = g - C*u
		B.multAdd(-1., q, t, r); // r = t - B*q = g - B*q - C*u
		Bt.multAdd(-1., u, f, qr); // qr = f - Bt*u
		solver.solve(A, qr, q); // q = A\qr

		for (iter.setFirst(); !iter.converged(r); iter.next()) {
			C.multAdd(-1., u, g, t); // t = g - C*u
			B.multAdd(-1., q, t, r); // r = g - B*q - C*u
			Bt.mult(r, p); // p = Bt*r
			solver.solve(A, p, h); // h = A\p

			double alpha = r.dot(r) / p.dot(h);
			u.add(-alpha, r); // u = u - alpha*r
			q.add(alpha, h); // q = q + alpha*h
		}
	}

}
