/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import mt.ll.BLASkernel.UpLo;

import java.util.Iterator;

/**
 * Lower symmetric dense matrix. It has the same storage layout as the
 * {@link mt.DenseMatrix DenseMatrix}, but only refers to elements below or
 * on the main diagonal. The remaining elements are never accessed nor changed,
 * and is known only by symmetry.
 */
public class LowerSymmDenseMatrix extends AbstractSymmDenseMatrix {

	private static final long serialVersionUID = 4004930764154605757L;

	/**
	 * Constructor for LowerSymmDenseMatrix
	 * 
	 * @param n
	 *            Size of the matrix. Since the matrix must be square, this
	 *            equals both the number of rows and columns
	 */
	public LowerSymmDenseMatrix(int n) {
		super(n, UpLo.Lower);
	}

	/**
	 * Constructor for LowerSymmDenseMatrix
	 * 
	 * @param A
	 *            Matrix to copy. It must be a square matrix, and only the
	 *            lower triangular part is copied
	 */
	public LowerSymmDenseMatrix(Matrix A) {
		this(A, true);
	}

	/**
	 * Constructor for LowerSymmDenseMatrix
	 * 
	 * @param A
	 *            Matrix to copy. It must be a square matrix, and only the
	 *            lower triangular part is copied
	 * @param deep
	 *            If false, a shallow copy is made. In that case, <code>A</code>
	 *            must be a dense matrix
	 */
	public LowerSymmDenseMatrix(Matrix A, boolean deep) {
		super(A, deep, UpLo.Lower);
	}

	public void add(int row, int column, double value) {
		if (column <= row)
			super.add(row, column, value);
	}

	public double get(int row, int column) {
		if (column > row)
			return super.get(column, row);
		return super.get(row, column);
	}

	public void set(int row, int column, double value) {
		if (column <= row)
			super.set(row, column, value);
	}

	//public LowerSymmDenseMatrix copy() {
	public Matrix copy() {
		return new LowerSymmDenseMatrix(this);
	}

	void copy(Matrix A) {
		/*for (MatrixEntry e : A)
			if (e.row() >= e.column())
				set(e.row(), e.column(), e.get());*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			if (e.row() >= e.column())
				set(e.row(), e.column(), e.get());
		}
	}

}
