/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.Matrix;
import mt.Vector;

/**
 * Givens plane rotation
 */
public class GivensRotation {

	/**
	 * Cosine and sine of the rotation
	 */
	private double c, s;

	/**
	 * Constructs an empty Givens plane rotation
	 */
	public GivensRotation() {
		// void
	}

	/**
	 * Constructs a Givens plane rotation
	 * 
	 * @param dx
	 * @param dy
	 */
	public GivensRotation(double dx, double dy) {
		double roe = Math.abs(dx) > Math.abs(dy) ? dx : dy;

		double scale = Math.abs(dx) + Math.abs(dy), r = 0;
		if (scale != 0.) {
			double dxs = dx / scale;
			double dys = dy / scale;
			r = scale * Math.sqrt(dxs * dxs + dys * dys);
			if (roe < 0.)
				r *= -1;
			c = dx / r;
			s = dy / r;
		} else {
			c = 1;
			s = 0;
			r = 0;
		}
	}

	/**
	 * Applies the Givens rotation
	 * 
	 * @param H
	 *            Matrix to apply to
	 * @param i1
	 *            Row index of first element
	 * @param j1
	 *            Column index of first element
	 * @param i2
	 *            Row index of second element
	 * @param j2
	 *            Column index of second element
	 */
	public void apply(Matrix H, int i1, int j1, int i2, int j2) {
		double temp = c * H.get(i1, j1) + s * H.get(i2, j2);
		H.set(i2, j2, -s * H.get(i1, j1) + c * H.get(i2, j2));
		H.set(i1, j1, temp);
	}

	/**
	 * Applies the Givens rotation
	 * 
	 * @param x
	 *            Vector to apply to
	 * @param i1
	 *            Index of first element
	 * @param i2
	 *            Index of second element
	 */
	public void apply(Vector x, int i1, int i2) {
		double temp = c * x.get(i1) + s * x.get(i2);
		x.set(i2, -s * x.get(i1) + c * x.get(i2));
		x.set(i1, temp);
	}

	/**
	 * Applies the Givens rotation
	 * 
	 * @param x
	 * @param y
	 */
	public void apply(Vector x, Vector y) {
		assert x.size() == y.size() : "x.size() != y.size()";
		for (int i = 0; i < x.size(); ++i) {
			double temp = c * x.get(i) + s * y.get(i);
			x.set(i, -s * x.get(i) + c * y.get(i));
			y.set(i, temp);
		}
	}

}
