/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import java.io.Serializable;

import mt.DenseMatrix;
import mt.ll.LAPACKkernel.JobEig;

/**
 * Eigenvalue decomposition. To compute a decomposition, use the
 * {@link mt.fact.EigenvalueComputer EigenvalueComputer}.
 */
public class EigenvalueDecomposition implements Serializable {

	private static final long serialVersionUID = -7013569148894802358L;

	/**
	 * Contains the real and imaginary parts of the eigenvalues
	 */
	private double[] Wr, Wi;

	/**
	 * Contains the left and the right eigenvectors
	 */
	private DenseMatrix Vl, Vr;

	/**
	 * Constructor for EigenvalueDecomposition
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param jobLeft
	 *            Whether or not to store the left eigenvectors
	 * @param jobRight
	 *            Whether or not to store the left eigenvectors
	 */
	public EigenvalueDecomposition(int n, JobEig jobLeft, JobEig jobRight) {
		Wr = new double[n];
		Wi = new double[n];

		if (jobLeft == JobEig.All)
			Vl = new DenseMatrix(n, n);
		if (jobRight == JobEig.All)
			Vr = new DenseMatrix(n, n);
	}

	/**
	 * Constructor for EigenvalueDecomposition
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param job
	 *            Whether or not to store the left and right eigenvectors
	 */
	public EigenvalueDecomposition(int n, JobEig job) {
		this(n, job, job);
	}

	/**
	 * Constructor for EigenvalueDecomposition. Stores all eigenvectors
	 * 
	 * @param n
	 *            Size of the matrix
	 */
	public EigenvalueDecomposition(int n) {
		this(n, JobEig.All);
	}

	/**
	 * Constructor for EigenvalueDecomposition. Stores all eigenvectors
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param job
	 *            True to store the eigenvectors in addition to eigenvalues,
	 *            else just the eigenvalues
	 */
	public EigenvalueDecomposition(int n, boolean job) {
		this(n, job ? JobEig.All : JobEig.Eigenvalues);
	}

	public DenseMatrix getLeftEigenvectors() {
		return Vl;
	}

	public DenseMatrix getRightEigenvectors() {
		return Vr;
	}

	public double[] getRealEigenvalues() {
		return Wr;
	}

	public double[] getImaginaryEigenvalues() {
		return Wi;
	}

	public boolean hasLeftEigenvectors() {
		return Vl != null;
	}

	public boolean hasRightEigenvectors() {
		return Vr != null;
	}

}
