/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import java.io.Serializable;
import java.util.Arrays;

/**
 * Dense vector. Stored by a <code>double[]</code> array of the same length
 * as the vector itself.
 */
public class DenseVector extends AbstractVector implements Serializable {

	private static final long serialVersionUID = -5039888671663863418L;

	/**
	 * Vector data
	 */
	private double[] data;

	/**
	 * Constructor for DenseVector
	 * 
	 * @param size
	 *            Size of the vector
	 */
	public DenseVector(int size) {
		super(size);
		data = new double[size];
	}

	/**
	 * Constructor for DenseVector
	 * 
	 * @param x
	 *            Copies contents from this vector. A deep copy is made
	 */
	public DenseVector(Vector x) {
		this(x, true);
	}

	/**
	 * Constructor for DenseVector
	 * 
	 * @param x
	 *            Copies contents from this vector
	 * @param deep
	 *            True for a deep copy. For a shallow copy, <code>x</code>
	 *            must be a <code>DenseVector</code>
	 */
	public DenseVector(Vector x, boolean deep) {
		super(x);

		if (deep) {
			data = new double[size];
			set(x);
		} else
			data = ((DenseVector) x).getData();
	}

	/**
	 * Constructor for DenseVector
	 * 
	 * @param x
	 *            Copies contents from this array
	 * @param deep
	 *            True for a deep copy. For a shallow copy, <code>x</code> is
	 *            aliased with the internal storage
	 */
	public DenseVector(double[] x, boolean deep) {
		super(x.length);

		if (deep)
			data = (double[]) x.clone();
		else
			data = x;
	}

	/**
	 * Constructor for DenseVector
	 * 
	 * @param x
	 *            Copies contents from this array in a deep copy
	 */
	public DenseVector(double[] x) {
		this(x, true);
	}

	/**
	 * Returns the internal vector contents. The array indices correspond to
	 * the vector indices
	 */
	public double[] getData() {
		return data;
	}

	public void add(int index, double value) {
		check(index);
		data[index] += value;
	}

	public void add(int[] index, double[] values) {
		check(index, values);

		for (int i = 0; i < index.length; ++i)
			data[index[i]] += values[i];
	}

	public double get(int index) {
		check(index);
		return data[index];
	}

	public double[] get(int[] index, double[] values) {
		check(index, values);

		for (int i = 0; i < index.length; ++i)
			values[i] = data[index[i]];
		return values;
	}

	public void set(int index, double value) {
		check(index);
		data[index] = value;
	}

	public void set(int[] index, double[] values) {
		check(index, values);

		for (int i = 0; i < index.length; ++i)
			data[index[i]] = values[i];
	}

	//public DenseVector copy() {
	public Vector copy() {
		return new DenseVector(this);
	}

	//public DenseVector zero() {
	public Vector zero() {
		Arrays.fill(data, 0);
		return this;
	}

	//public DenseVector scale(double alpha) {
	public Vector scale(double alpha) {
		for (int i = 0; i < size; ++i)
			data[i] *= alpha;
		return this;
	}

	public Vector set(Vector y) {
		if (this == y)
			return this;
		if (!(y instanceof DenseVector))
			return super.set(y);

		checkSize(y);

		double[] yd = ((DenseVector) y).getData();
		System.arraycopy(yd, 0, data, 0, size);

		return this;
	}

	public double dot(Vector y) {
		if (!(y instanceof DenseVector))
			return super.dot(y);

		checkSize(y);

		double[] yd = ((DenseVector) y).getData();

		double dot = 0.;
		for (int i = 0; i < size; ++i)
			dot += data[i] * yd[i];
		return dot;
	}

	protected double norm1() {
		double sum = 0;
		for (int i = 0; i < size; ++i)
			sum += Math.abs(data[i]);
		return sum;
	}

	protected double norm2() {
		double norm = 0;
		for (int i = 0; i < size; ++i)
			norm += data[i] * data[i];
		return Math.sqrt(norm);
	}

	protected double norm2_robust() {
		double scale = 0, ssq = 1;
		for (int i = 0; i < size; ++i) {
			if (data[i] != 0) {
				double absxi = Math.abs(data[i]);
				if (scale < absxi) {
					ssq = 1 + ssq * Math.pow(scale / absxi, 2);
					scale = absxi;
				} else
					ssq = ssq + Math.pow(absxi / scale, 2);
			}
		}
		return scale * Math.sqrt(ssq);
	}

	protected double normInf() {
		double max = 0;
		for (int i = 0; i < size; ++i)
			max = Math.max(Math.abs(data[i]), max);
		return max;
	}

	public Vector set(double alpha, Vector y, double beta, Vector z) {
		if (!(y instanceof DenseVector) || !(z instanceof DenseVector))
			return super.set(alpha, y, beta, z);

		checkSet(y, z);

		double[] yd = ((DenseVector) y).getData(),
			zd = ((DenseVector) z).getData();

		for (int i = 0; i < size; i++)
			data[i] = alpha * yd[i] + beta * zd[i];

		return this;
	}

	//public DenseVector set(double alpha) {
	public Vector set(double alpha) {
		Arrays.fill(data, alpha);
		return this;
	}

	public Vector set(double alpha, Vector y) {
		if (!(y instanceof DenseVector))
			return super.set(alpha, y);

		checkSize(y);

		double[] yd = ((DenseVector) y).getData();

		for (int i = 0; i < size; ++i)
			data[i] = alpha * yd[i];

		return this;
	}

}
