/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.LowerSymmBandMatrix;
import mt.Matrix;
import mt.UpperSymmBandMatrix;
import mt.ll.BLASkernel.UpLo;
import mt.ll.Interface;
import mt.ll.LAPACKkernel.JobEig;

/**
 * Computes eigenvalues of symmetrical, banded matrices
 */
public class SymmBandEigenvalueComputer {

	/**
	 * Double work array
	 */
	private double[] work;

	/**
	 * Integer work array
	 */
	private int[] iwork;

	/**
	 * Size of the matrix
	 */
	private int n;

	/**
	 * Upper or lower part stored
	 */
	private UpLo uplo;

	/**
	 * Job to do
	 */
	private JobEig job;

	/**
	 * Creates a new eigenvalue computer for symmetrical, banded matrices.
	 * Computes all eigenvalues and eigenvectors
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param uplo
	 *            Lower or upper part stored?
	 */
	public SymmBandEigenvalueComputer(int n, UpLo uplo) {
		this(n, uplo, JobEig.All);
	}

	/**
	 * Creates a new eigenvalue computer for symmetrical, banded matrices
	 * 
	 * @param n
	 *            Size of the matrix
	 * @param uplo
	 *            Lower or upper part stored?
	 * @param job
	 *            The type of computation to perform
	 */
	public SymmBandEigenvalueComputer(int n, UpLo uplo, JobEig job) {
		this.n = n;
		this.uplo = uplo;
		this.job = job;

		// Find the needed workspace
		work = new double[1];
		iwork = new int[1];
		int info =
			Interface.lapack().sbevd(
				job,
				uplo,
				n,
				0,
				new double[0],
				new double[0],
				new double[0],
				work,
				-1,
				iwork,
				-1);

		// Allocate workspace
		int lwork = 0, liwork = 0;
		if (info != 0) {
			if (job == JobEig.All) {
				lwork = 1 + 6 * n + 2 * n * n;
				liwork = 3 + 5 * n;
			} else {
				lwork = 2 * n;
				liwork = 1;
			}
		} else {
			lwork = (int) work[0];
			liwork = iwork[0];
		}
		lwork = Math.max(1, lwork);
		liwork = Math.max(1, liwork);
		work = new double[lwork];
		iwork = new int[liwork];
	}

	/**
	 * Convience method for computing the full eigenvalue decomposition of the
	 * given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. The upper triangular part is extracted,
	 *            and the matrix is not modified
	 * @param kd
	 *            Number of diagonals to extract
	 * @return
	 *            Newly allocated decomposition
	 * @throws NotConvergedException
	 */
	public static SymmEigenvalueDecomposition factorize(Matrix A, int kd)
		throws NotConvergedException {
		return new SymmBandEigenvalueComputer(A.numRows(), UpLo.Upper).factor(
			new UpperSymmBandMatrix(A, kd));
	}

	/**
	 * Computes the eigenvalue decomposition of the given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Overwritten on return
	 * @return
	 *            Newly allocated decomposition
	 * @throws NotConvergedException
	 */
	public SymmEigenvalueDecomposition factor(LowerSymmBandMatrix A)
		throws NotConvergedException {
		return factor(A, new SymmEigenvalueDecomposition(n, job));
	}

	/**
	 * Computes the eigenvalue decomposition of the given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Overwritten on return
	 * @return
	 *            Newly allocated decomposition
	 * @throws NotConvergedException
	 */
	public SymmEigenvalueDecomposition factor(UpperSymmBandMatrix A)
		throws NotConvergedException {
		return factor(A, new SymmEigenvalueDecomposition(n, job));
	}

	/**
	 * Computes the eigenvalue decomposition of the given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Overwritten on return
	 * @param evd
	 *            Factorization is written here
	 * @return
	 *            evd
	 * @throws NotConvergedException
	 */
	public SymmEigenvalueDecomposition factor(
		LowerSymmBandMatrix A,
		SymmEigenvalueDecomposition evd)
		throws NotConvergedException {
		if (uplo != UpLo.Lower)
			throw new IllegalArgumentException("Eigenvalue computer configured for lower-symmetrical matrices");

		return factor(A, A.getData(), A.numSubDiagonals(), evd);
	}

	/**
	 * Computes the eigenvalue decomposition of the given matrix
	 * 
	 * @param A
	 *            Matrix to factorize. Overwritten on return
	 * @param evd
	 *            Factorization is written here
	 * @return
	 *            evd
	 * @throws NotConvergedException
	 */
	public SymmEigenvalueDecomposition factor(
		UpperSymmBandMatrix A,
		SymmEigenvalueDecomposition evd)
		throws NotConvergedException {
		if (uplo != UpLo.Upper)
			throw new IllegalArgumentException("Eigenvalue computer configured for upper-symmetrical matrices");

		return factor(A, A.getData(), A.numSuperDiagonals(), evd);
	}

	private SymmEigenvalueDecomposition factor(
		Matrix A,
		double[] data,
		int kd,
		SymmEigenvalueDecomposition evd)
		throws NotConvergedException {
		if (A.numRows() != n)
			throw new IllegalArgumentException("A.numRows() != n");
		else if (evd.getEigenvalues().length != n)
			throw new IllegalArgumentException("evd.getEigenvalues().length != n");
		else if (job == JobEig.All) {
			if (!evd.hasEigenvectors())
				throw new IllegalArgumentException("Eigenvalue decomposition does not store eigenvectors");
			else if (evd.getEigenvectors().numRows() != n)
				throw new IllegalArgumentException("evd.getEigenvectors().numRows() != n");
			else if (evd.getEigenvectors().numColumns() != n)
				throw new IllegalArgumentException("evd.getEigenvectors().numColumns() != n");
		}

		int info =
			Interface.lapack().sbevd(
				job,
				uplo,
				n,
				kd,
				data,
				evd.getEigenvalues(),
				job == JobEig.All
					? evd.getEigenvectors().getData()
					: new double[0],
				work,
				work.length,
				iwork,
				iwork.length);

		if (info > 0)
			throw new NotConvergedException(
				NotConvergedException.Reason.Iterations);
		else if (info < 0)
			throw new IllegalArgumentException();

		return evd;
	}

}
