/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.DenseMatrix;
import mt.Matrix;
import mt.UpperTriangDenseMatrix;
import mt.ll.Interface;
import mt.ll.BLASkernel.Side;
import mt.ll.BLASkernel.Transpose;

/**
 * Computers QR decompositions. Multiple threads should not share a single QR
 * instance nor the produced factorizations. This is due to workspace sharing
 * issues
 */
public class QRComputer {

	/**
	 * Work arrays. These are shared between all generated orthogonal
	 * factorizations
	 */
	private double[] work, workGen, workMult, workMultTrans;

	/**
	 * Constructor for QR
	 * 
	 * @param m
	 *            Number of rows
	 * @param n
	 *            Number of columns
	 */
	public QRComputer(int m, int n) {

		int info, lwork;

		// Query optimal workspace. First for computing the factorization
		{
			work = new double[1];
			info =
				Interface.lapack().geqrf(
					m,
					n,
					new double[0],
					new double[0],
					work,
					-1);

			if (info != 0)
				lwork = n;
			else
				lwork = (int) work[0];
			lwork = Math.max(1, lwork);
			work = new double[lwork];
		}

		// Workspace needed for generating an explicit orthogonal matrix
		{
			workGen = new double[1];
			info =
				Interface.lapack().orgqr(
					m,
					n,
					n,
					new double[0],
					new double[0],
					workGen,
					-1);

			if (info != 0)
				lwork = n;
			else
				lwork = (int) workGen[0];
			lwork = Math.max(1, lwork);
			workGen = new double[lwork];
		}

		// Workspace needed for multiplying the orthogonal matrix
		// Non-transpose case
		{
			workMult = new double[1];
			info =
				Interface.lapack().ormqr(
					Side.Left,
					Transpose.NoTranspose,
					m,
					n,
					n,
					new double[0],
					new double[0],
					new double[0],
					workMult,
					-1);

			if (info != 0)
				lwork = n;
			else
				lwork = (int) workMult[0];
			lwork = Math.max(1, lwork);
			workMult = new double[lwork];
		}

		// Transpose case
		{
			workMultTrans = new double[1];
			info =
				Interface.lapack().ormqr(
					Side.Left,
					Transpose.Transpose,
					m,
					n,
					n,
					new double[0],
					new double[0],
					new double[0],
					workMultTrans,
					-1);

			if (info != 0)
				lwork = n;
			else
				lwork = (int) workMultTrans[0];
			lwork = Math.max(1, lwork);
			workMultTrans = new double[lwork];
		}

	}

	/**
	 * Convience method to compute a QR decomposition
	 * 
	 * @param A
	 *            Matrix to decompose. Not modified
	 * @return
	 *            Newly allocated decomposition
	 */
	public static OrthogonalDecomposition factorize(Matrix A) {
		return new QRComputer(A.numRows(), A.numColumns()).factor(A);
	}

	/**
	 * Computes a QR decomposition
	 * 
	 * @param A
	 *            Matrix to decompose. Not modified
	 * @return
	 *            Newly allocated decomposition
	 */
	public OrthogonalDecomposition factor(Matrix A) {
		return factor(
			A,
			new OrthogonalDecomposition(A.numRows(), A.numColumns()));
	}

	/**
	 * Computes a QR decomposition
	 * 
	 * @param A
	 *            Matrix to decompose. Not modified
	 * @param od
	 *            Decomposition is written here
	 * @return
	 *            The passed decomposition
	 */
	public OrthogonalDecomposition factor(
		Matrix A,
		OrthogonalDecomposition od) {
		if (od.getH().numRows() != A.numRows())
			throw new IllegalArgumentException("od.getH().numRows() != A.numRows()");
		else if (od.getH().numColumns() != A.numColumns())
			throw new IllegalArgumentException("od.getH().numColumns() != A.numColumns()");
		else if (od.getTau().length != Math.min(A.numRows(), A.numColumns()))
			throw new IllegalArgumentException("od.getTau().length != Math.min(A.numRows(),A.numColumns())");

		od.getH().set(A);

		int info =
			Interface.lapack().geqrf(
				A.numRows(),
				A.numColumns(),
				od.getH().getData(),
				od.getTau(),
				work,
				work.length);

		if (info < 0)
			throw new IllegalArgumentException();

		od.setQ(
			new OrthQRMatrix(
				od.getH(),
				od.getTau(),
				workGen,
				workMult,
				workMultTrans));
		od.setR(new UpperTriangDenseMatrix(od.getH()));

		return od;
	}

	private class OrthQRMatrix extends OrthMatrix {

		private static final long serialVersionUID = -2709840967972191375L;

		private double[] workGen, workMult, workMultTrans;

		public OrthQRMatrix(
			DenseMatrix H,
			double[] tau,
			double[] workGen,
			double[] workMult,
			double[] workMultTrans) {
			super(H, tau);
			this.workGen = workGen;
			this.workMult = workMult;
			this.workMultTrans = workMultTrans;
		}

		public DenseMatrix generate() {
			DenseMatrix Q = new DenseMatrix(H);
			int m = H.numRows(), n = H.numColumns();

			int info =
				Interface.lapack().orgqr(
					m,
					n,
					n,
					Q.getData(),
					tau,
					workGen,
					workGen.length);

			if (info < 0)
				throw new IllegalArgumentException();

			return Q;
		}

		void mult(DenseMatrix D, Transpose trans) {
			double[] Dd = D.getData();
			int m = H.numRows(), n = H.numColumns();

			double[] work =
				trans == Transpose.NoTranspose ? workMult : workMultTrans;

			int info =
				Interface.lapack().ormqr(
					Side.Left,
					trans,
					m,
					n,
					n,
					H.getData(),
					tau,
					Dd,
					work,
					work.length);

			if (info < 0)
				throw new IllegalArgumentException();
		}
	}

}
