/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import java.util.Arrays;

import mt.ll.BLASkernel.Transpose;
import mt.ll.Interface;

/**
 * Banded matrix. The banded matrix is a useful sparse structure for many kinds
 * of direct computations, however it should only be used if the band is
 * sufficiently narrow. as wide bands actually wastes both memory and compute
 * time. The matrix
 * <p>
 * <table border="1">
 * <tr>
 * <td>a <sub>11 </sub></td>
 * <td>a <sub>12 </sub></td>
 * <td>&nbsp;</td>
 * <td>&nbsp;</td>
 * <td>&nbsp;</td>
 * </tr>
 * <tr>
 * <td>a <sub>21 </sub></td>
 * <td>a <sub>22 </sub></td>
 * <td>a <sub>23 </sub></td>
 * <td>&nbsp;</td>
 * <td>&nbsp;</td>
 * </tr>
 * <tr>
 * <td>a <sub>31 </sub></td>
 * <td>a <sub>32 </sub></td>
 * <td>a <sub>33 </sub></td>
 * <td>a <sub>34 </sub></td>
 * <td>&nbsp;</td>
 * </tr>
 * <tr>
 * <td>&nbsp;</td>
 * <td>a <sub>42 </sub></td>
 * <td>a <sub>43 </sub></td>
 * <td>a <sub>44 </sub></td>
 * <td>a <sub>45 </sub></td>
 * </tr>
 * <tr>
 * <td>&nbsp;</td>
 * <td>&nbsp;</td>
 * <td>a <sub>53 </sub></td>
 * <td>a <sub>54 </sub></td>
 * <td>a <sub>55 </sub></td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * has two lower diagonals and one upper diagonal. It will be stored in the
 * array
 * </p>
 * <p>
 * <table border="1">
 * <tr>
 * <td>&nbsp;</td>
 * <td>a <sub>11 </sub></td>
 * <td>a <sub>21 </sub></td>
 * <td>a <sub>31 </sub></td>
 * <td>a <sub>21 </sub></td>
 * <td>a <sub>22 </sub></td>
 * <td>a <sub>32 </sub></td>
 * <td>a <sub>42 </sub></td>
 * <td>a <sub>23 </sub></td>
 * <td>a <sub>33 </sub></td>
 * <td>a <sub>43 </sub></td>
 * <td>a <sub>53 </sub></td>
 * <td>a <sub>34 </sub></td>
 * <td>a <sub>44 </sub></td>
 * <td>a <sub>54 </sub></td>
 * <td>&nbsp;</td>
 * <td>a <sub>45 </sub></td>
 * <td>a <sub>55 </sub></td>
 * <td>&nbsp;</td>
 * <td>&nbsp;</td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * Empty cells are allocated, but never referenced.
 * </p>
 */
public class BandMatrix extends AbstractBandMatrix {

	private static final long serialVersionUID = -188355296366150775L;

	/**
	 * Constructor for BandMatrix
	 * 
	 * @param n
	 *            Size of the matrix. Since the matrix must be square, this
	 *            equals both the number of rows and columns
	 * @param kl
	 *            Number of bands above the main diagonal (superdiagonals)
	 * @param ku
	 *            Number of bands below the main diagonal (subdiagonals)
	 */
	public BandMatrix(int n, int kl, int ku) {
		super(n, kl, ku);
	}

	/**
	 * Constructor for BandMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. Only the parts of <code>A</code>
	 *            that lie within the allocated band are copied over, the rest
	 *            is ignored
	 * @param kl
	 *            Number of bands above the main diagonal (superdiagonals)
	 * @param ku
	 *            Number of bands below the main diagonal (subdiagonals)
	 */
	public BandMatrix(Matrix A, int kl, int ku) {
		super(A, kl, ku);
	}

	/**
	 * Constructor for BandMatrix
	 * 
	 * @param A
	 *            Matrix to copy contents from. Only the parts of <code>A</code>
	 *            that lie within the allocated band are copied over, the rest
	 *            is ignored
	 * @param kl
	 *            Number of bands above the main diagonal (superdiagonals)
	 * @param ku
	 *            Number of bands below the main diagonal (subdiagonals)
	 * @param deep
	 *            True for a deep copy. For shallow copies, <code>A</code>
	 *            must be a banded matrix
	 */
	public BandMatrix(Matrix A, int kl, int ku, boolean deep) {
		super(A, kl, ku, deep);
	}

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.multAdd(alpha, x, beta, y, z);

		checkMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		z.set(y);

		Interface.blas().gbmv(
			Transpose.NoTranspose,
			numRows,
			numColumns,
			kl,
			ku,
			alpha,
			data,
			kl + ku + 1,
			xd,
			beta,
			zd);

		return z;
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
			return super.transMultAdd(alpha, x, beta, y, z);

		checkTransMultAdd(x, y, z);

		double[] xd = ((DenseVector) x).getData(),
			zd = ((DenseVector) z).getData();

		z.set(y);

		Interface.blas().gbmv(
			Transpose.Transpose,
			numRows,
			numColumns,
			kl,
			ku,
			alpha,
			data,
			kl + ku + 1,
			xd,
			beta,
			zd);

		return z;
	}

	//public BandMatrix copy() {
	public Matrix copy() {
		return new BandMatrix(this, kl, ku);
	}

	public Matrix solve(Matrix B, Matrix X) {
		if (!(X instanceof DenseMatrix))
			throw new UnsupportedOperationException("X must be a DenseMatrix");

		checkSolve(B, X);

		double[] Xd = ((DenseMatrix) X).getData();

		X.set(B);

		// Allocate factorization matrix. The factorization matrix will be
		// large enough to accomodate any pivots
		BandMatrix Af = new BandMatrix(this, kl, ku + kl);
		int[] ipiv = new int[numRows];

		int info =
			Interface.lapack().gbsv(
				numRows,
				kl,
				ku,
				X.numColumns(),
				Af.getData(),
				ipiv,
				Xd);

		if (info > 0)
			throw new MatrixSingularException();
		else if (info < 0)
			throw new IllegalArgumentException();

		return X;
	}

	public Vector solve(Vector b, Vector x) {
		DenseMatrix B = new DenseMatrix(b, false),
			X = new DenseMatrix(x, false);
		solve(B, X);
		return x;
	}

	public Matrix zero() {
		Arrays.fill(data, 0);
		return this;
	}

}
