/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.mixed;

import mt.Matrix;
import mt.Vector;
import smt.iter.IterationMonitor;
import smt.iter.IterativeSolver;
import smt.iter.IterativeSolverNotConvergedException;

/**
 * Solver for mixed problems
 */
public interface MixedSolver {

	/**
	 * Solves the given mixed problem
	 * 
	 * @param A
	 *            Square matrix with <i>m</i> rows and columns
	 * @param B
	 *            Matrix with <i>n</i> rows and <i>m</i> columns
	 * @param Bt
	 *            Matrix with <i>m</i> rows and <i>n</i> columns
	 * @param C
	 *            Square matrix with <i>n</i> rows and columns
	 * @param q
	 *            Vector of size <i>m</i>. Overwritten on exit
	 * @param u
	 *            Vector of size <i>n</i>. Overwritten on exit
	 * @param f
	 *            Vector of size <i>m</i>
	 * @param g
	 *            Vector of size <i>n</i>
	 */
	void solve(
		Matrix A,
		Matrix B,
		Matrix Bt,
		Matrix C,
		Vector q,
		Vector u,
		Vector f,
		Vector g)
		throws IterativeSolverNotConvergedException;

	/**
	 * Solves the given mixed problem. Sets <code>Bt=B<sup>T</sup></code>
	 * 
	 * @param A
	 *            Square matrix with <i>m</i> rows and columns
	 * @param B
	 *            Matrix with <i>n</i> rows and <i>m</i> columns
	 * @param C
	 *            Square matrix with <i>n</i> rows and columns
	 * @param q
	 *            Vector of size <i>m</i>. Overwritten on exit
	 * @param u
	 *            Vector of size <i>n</i>. Overwritten on exit
	 * @param f
	 *            Vector of size <i>m</i>
	 * @param g
	 *            Vector of size <i>n</i>
	 */
	void solve(
		Matrix A,
		Matrix B,
		Matrix C,
		Vector q,
		Vector u,
		Vector f,
		Vector g)
		throws IterativeSolverNotConvergedException;

	/**
	 * Sets the solver to use internally
	 * 
	 * @param solver
	 *            IterativeSolver to use
	 */
	void setSubSolver(IterativeSolver solver);

	/**
	 * Returns internal solver
	 * 
	 * @return The solver currently in use
	 */
	IterativeSolver getSubSolver();

	/**
	 * Sets the iteration monitor
	 * 
	 * @param iter
	 *            Iteration monitor
	 */
	void setIterationMonitor(IterationMonitor iter);

	/**
	 * Gets the iteration monitor
	 * 
	 * @return Current iteration monitor
	 */
	IterationMonitor getIterationMonitor();

}
