/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.util;

import mt.AbstractMatrix;
import mt.AbstractVector;
import mt.DenseMatrix;
import mt.DenseVector;
import mt.Matrix;
import mt.MatrixEntry;
import mt.Vector;
import mt.VectorEntry;

import java.util.Iterator;

/**
 * Creates matrices and vectors
 */
public final class Matrices {

	private Matrices() {
		// No need to instantiate
	}

	/**
	 * Returns an array of arrays containing a copy of the given matrix.
	 * Each array contains one row.
	 */
	public static double[][] getArray(Matrix A) {
		double[][] Ad = new double[A.numRows()][A.numColumns()];
		/*for (MatrixEntry e : A)
			Ad[e.row()][e.column()] = e.get();*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			Ad[e.row()][e.column()] = e.get();
		}
		return Ad;
	}

	/**
	 * Returns a dense array containing a copy of the given vector
	 */
	public static double[] getArray(Vector x) {
		double[] xd = new double[x.size()];
		/*for (VectorEntry e : x)
			xd[e.index()] = e.get();*/
		VectorEntry e;
		Iterator iter = x.iterator();
		while(iter.hasNext()) {
			e = (VectorEntry) iter.next();
			xd[e.index()] = e.get();
		}
		return xd;
	}

	/**
	 * Returns the identity matrix of the given size
	 * 
	 * @param size
	 *            Number of rows/columns of the matrix
	 * @return    Matrix of the given size, with ones on the main diagonal
	 */
	public static Matrix identity(int size) {
		return new DenseMatrix(size, size).addDiagonal(1);
	}

	/**
	 * Creates a random vector. Numbers are drawn from a uniform distribution
	 * between 0 and 1
	 * 
	 * @param size
	 *            Size of the vector
	 */
	public static Vector random(int size) {
		return random(new DenseVector(size));
	}

	/**
	 * Populates a vector with random numbers drawn from a uniform distribution
	 * between 0 and 1
	 * 
	 * @param x
	 *            Vector to populate
	 */
	public static Vector random(Vector x) {
		for (int i = 0; i < x.size(); ++i)
			x.set(i, Math.random());
		return x;
	}

	/**
	 * Creates a random matrix. Numbers are drawn from a uniform distribution
	 * between 0 and 1
	 * 
	 * @param numRows
	*            Number of rows
	 * @param numColumns
	 *            Number of columns
	 */
	public static Matrix random(int numRows, int numColumns) {
		return random(new DenseMatrix(numRows, numColumns));
	}

	/**
	 * Populates a matrix with random numbers drawn from a uniform distribution
	 * between 0 and 1
	 * 
	 * @param A
	 *            Matrix to populate
	 */
	public static Matrix random(Matrix A) {
		for (int j = 0; j < A.numColumns(); ++j)
			for (int i = 0; i < A.numRows(); ++i)
				A.set(i, j, Math.random());
		return A;
	}

	/**
	 * Returns a synchronized vector which wraps the given vector. Only the
	 * <code>set(int, double)</code> and <code>add(int, double)</code>
	 * methods and their blocked versions are synchronized.
	 * <p>
	 * <b>Note:</b> Do not use the wrapped vector for any operations besides
	 * matrix assembly, as these operations may be very slow.
	 * 
	 * @param x
	 *            Vector to be wrapped
	 * @return A thin wrapper around <code>x</code>
	 */
	public static Vector synchronizedVector(Vector x) {
		return new SynchronizedVector(x);
	}

	/**
	 * Returns a synchronized matrix which wraps the given matrix. Only the
	 * <code>set(int, int, double)</code> and <code>add(int, int, double)</code>
	 * methods and their blocked versions are synchronized.
	 * <p>
	 * <b>Note:</b> Do not use the wrapped matrix for any operations besides
	 * matrix assembly, as these operations may be very slow.
	 * 
	 * @param A
	 *            Matrix to be wrapped
	 * @return A thin wrapper around <code>A</code>
	 */
	public static Matrix synchronizedMatrix(Matrix A) {
		return new SynchronizedMatrix(A);
	}

	/**
	 * Returns a synchronized matrix which wraps the given matrix. Only the
	 * <code>set(int, int, double)</code> and <code>add(int, int, double)</code>
	 * methods and their blocked versions are synchronized.
	 * <p>
	 * The locking provided is finer than the locking of the whole matrix, as
	 * different threads can access different rows simultaneous, while only one
	 * thread can access a given row at a time. Use this for row-major
	 * matrices, <i>not</i> for column-major matrices.
	 * <p>
	 * <b>Note:</b> Do not use the wrapped matrix for any operations besides
	 * matrix assembly, as these operations may be very slow.
	 * 
	 * @param A
	 *            Matrix to be wrapped
	 * @return A thin wrapper around <code>A</code>. Individual rows are
	 *         locked
	 */
	public static Matrix synchronizedMatrixByRows(Matrix A) {
		return new SynchronizedRowMatrix(A);
	}

	/**
	 * Returns a synchronized matrix which wraps the given matrix. Only the
	 * <code>set(int, int, double)</code> and <code>add(int, int, double)</code>
	 * methods and their blocked versions are synchronized.
	 * <p>
	 * The locking provided is finer than the locking of the whole matrix, as
	 * different threads can access different columns simultaneous, while only
	 * one thread can access a given column at a time. Use this for
	 * column-major matrices, <i>not</i> for row-major matrices.
	 * <p>
	 * <b>Note:</b> Do not use the wrapped matrix for any operations besides
	 * matrix assembly, as these operations may be very slow.
	 * 
	 * @param A
	 *            Matrix to be wrapped
	 * @return A thin wrapper around <code>A</code>. Individual columns are
	 *         locked
	 */
	public static Matrix synchronizedMatrixByColumns(Matrix A) {
		return new SynchronizedColumnMatrix(A);
	}

	/**
	 * Returns a view into the given matrix. This view is only for easing
	 * some matrix-assembly cases, not for general use. To extract a more
	 * higher-performing and general matrix, create a copy of the submatrix.
	 * The result is a {@link mt.DenseMatrix DenseMatrix}.
	 * 
	 * @param A
	 *            Matrix to create view on
	 * @param row
	 *            Rows to access. Must be within the bounds of <code>A</code>
	 * @param column
	 *            Columns to access. Must be within the bounds of <code>A</code>
	 * @return
	 *            Submatrix of <code>A</code>. Changing it will change the
	 *            backing matrix
	 */
	public static Matrix getSubMatrix(Matrix A, int[] row, int[] column) {
		return new RefMatrix(A, row, column);
	}

	/**
	 * Returns a view into the given vector. This view is only for easing
	 * some vector-assembly cases, not for general use. To extract a more
	 * higher-performing and general vector, create a copy of the subvector.
	 * The result is a {@link mt.DenseVector DenseVector}.
	 * 
	 * @param x
	 *            Vector to create view on
	 * @param index
	 *            Indices to access. Must be within the bounds of <code>x</code>
	 * @return
	 *            Submatrix of <code>x</code>. Changing it will change the
	 *            backing matrix
	 */
	public static Vector getSubVector(Vector x, int[] index) {
		return new RefVector(x, index);
	}

	/**
	 * Matrix backed by another matrix. Used by <code>getSubMatrix</code>
	 */
	private static class RefMatrix extends AbstractMatrix {

		private Matrix A;

		private int[] row, column;

		public RefMatrix(Matrix A, int[] row, int[] column) {
			super(row.length, column.length);
			this.A = A;
			this.row = row;
			this.column = column;
		}

		public void add(int row, int column, double value) {
			A.add(this.row[row], this.column[column], value);
		}

		//public DenseMatrix copy() {
		public Matrix copy() {
			return new DenseMatrix(this);
		}

		public double get(int row, int column) {
			return A.get(this.row[row], this.column[column]);
		}

		public void set(int row, int column, double value) {
			A.set(this.row[row], this.column[column], value);
		}

	}

	/**
	 * Vector backed by another vector. Used by <code>getSubVector</code>
	 */
	private static class RefVector extends AbstractVector {

		private Vector x;

		private int[] index;

		public RefVector(Vector x, int[] index) {
			super(index.length);
			this.x = x;
			this.index = index;
		}

		public void add(int index, double value) {
			x.add(this.index[index], value);
		}

		//public DenseVector copy() {
		public Vector copy() {
			return new DenseVector(this);
		}

		public double get(int index) {
			return x.get(this.index[index]);
		}

		public void set(int index, double value) {
			x.set(this.index[index], value);
		}

	}

	/**
	 * Ensures correctness in the vector assembly. Since it extends the
	 * AbstractVector class, algebraic operations will be slow. It is not
	 * possible to implement Vector and delegate calls to the imbedded vector,
	 * since casting to the imbedded vector is not possible
	 */
	private static class SynchronizedVector extends AbstractVector {

		private Vector x;

		public SynchronizedVector(Vector x) {
			super(x);
			this.x = x;
		}

		public synchronized void add(int index, double value) {
			x.add(index, value);
		}

		public synchronized void add(int[] index, double[] values) {
			x.add(index, values);
		}

		public synchronized void set(int index, double value) {
			x.set(index, value);
		}

		public synchronized void set(int[] index, double[] values) {
			x.set(index, values);
		}

		public double get(int index) {
			return x.get(index);
		}

		public Vector copy() {
			return Matrices.synchronizedVector(x.copy());
		}

	}

	/**
	 * Ensures correctness in the matrix assembly. Since it extends the
	 * AbstractMatrix class, algebraic operations will be slow. It is not
	 * possible to implement Matrix and delegate calls to the imbedded matrix,
	 * since casting to the imbedded matrix is not possible
	 */
	private static class SynchronizedMatrix extends AbstractMatrix {

		private Matrix A;

		public SynchronizedMatrix(Matrix A) {
			super(A);
			this.A = A;
		}

		public synchronized void add(int row, int column, double value) {
			A.add(row, column, value);
		}

		public synchronized void add(
			int[] row,
			int[] column,
			double[][] values) {
			A.add(row, column, values);
		}

		public synchronized void set(int row, int column, double value) {
			A.set(row, column, value);
		}

		public synchronized void set(
			int[] row,
			int[] column,
			double[][] values) {
			A.set(row, column, values);
		}

		public double get(int row, int column) {
			return A.get(row, column);
		}

		public Matrix copy() {
			return Matrices.synchronizedMatrix(A.copy());
		}

	}

	/**
	 * Ensures correctness in the matrix assembly. Since it extends the
	 * AbstractMatrix class, algebraic operations will be slow. It is not
	 * possible to implement Matrix and delegate calls to the imbedded matrix,
	 * since casting to the imbedded matrix is not possible
	 * <p>
	 * Locks individual rows instead of the whole matrix
	 */
	private static class SynchronizedRowMatrix extends AbstractMatrix {

		private Matrix A;

		private Object[] lock;

		public SynchronizedRowMatrix(Matrix A) {
			super(A);
			this.A = A;
			lock = new Object[A.numRows()];
			for (int i = 0; i < lock.length; ++i)
				lock[i] = new Object();
		}

		public void add(int row, int column, double value) {
			synchronized (lock[row]) {
				A.add(row, column, value);
			}
		}

		public void add(int[] row, int[] column, double[][] values) {
			for (int i = 0; i < row.length; ++i)
				synchronized (lock[row[i]]) {
					for (int j = 0; j < column.length; ++j)
						A.add(row[i], column[j], values[i][j]);
				}
		}

		public void set(int row, int column, double value) {
			synchronized (lock[row]) {
				A.set(row, column, value);
			}
		}

		public void set(int[] row, int[] column, double[][] values) {
			for (int i = 0; i < row.length; ++i)
				synchronized (lock[row[i]]) {
					for (int j = 0; j < column.length; ++j)
						A.set(row[i], column[j], values[i][j]);
				}
		}

		public double get(int row, int column) {
			return A.get(row, column);
		}

		public Matrix copy() {
			return Matrices.synchronizedMatrixByRows(A.copy());
		}

	}

	/**
	 * Ensures correctness in the matrix assembly. Implements matrix instead of
	 * subclassing the abstract matrix in order to correctly delegate every
	 * method to possbly overridden method in the encapsulated matrix.
	 * <p>
	 * Locks individual columns instead of the whole matrix
	 */
	private static class SynchronizedColumnMatrix extends AbstractMatrix {

		private Matrix A;

		private Object[] lock;

		public SynchronizedColumnMatrix(Matrix A) {
			super(A);
			this.A = A;
			lock = new Object[A.numColumns()];
			for (int i = 0; i < lock.length; ++i)
				lock[i] = new Object();
		}

		public void add(int row, int column, double value) {
			synchronized (lock[column]) {
				A.add(row, column, value);
			}
		}

		public void add(int[] row, int[] column, double[][] values) {
			for (int j = 0; j < column.length; ++j)
				synchronized (lock[column[j]]) {
					for (int i = 0; i < row.length; ++i)
						A.add(row[i], column[j], values[i][j]);
				}
		}

		public void set(int row, int column, double value) {
			synchronized (lock[column]) {
				A.set(row, column, value);
			}
		}

		public void set(int[] row, int[] column, double[][] values) {
			for (int j = 0; j < column.length; ++j)
				synchronized (lock[column[j]]) {
					for (int i = 0; i < row.length; ++i)
						A.add(row[i], column[j], values[i][j]);
				}
		}

		public double get(int row, int column) {
			return A.get(row, column);
		}

		public Matrix copy() {
			return Matrices.synchronizedMatrixByColumns(A.copy());
		}

	}

}
