/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.test;

import mt.Matrix;
import mt.MatrixEntry;
import mt.MatrixNotSPDException;
import mt.MatrixSingularException;
import mt.Vector;
import mt.util.Matrices;
import junit.framework.TestCase;

import java.util.Iterator;

/**
 * Tests a matrix
 */
public abstract class MatrixTest extends TestCase {

	/**
	 * Matrix to test
	 */
	protected Matrix A;

	/**
	 * Jagged array version of A
	 */
	protected double[][] Ad;

	/**
	 * Matrix of the same size as A, dense and non-dense
	 */
	protected Matrix Bdense, B;

	/**
	 * Contents of B
	 */
	protected double[][] Bd;

	/**
	 * Vectors for multiplication and rank tests
	 */
	protected Vector xR,
		yR,
		zR,
		xC,
		yC,
		zC,
		xDenseR,
		yDenseR,
		zDenseR,
		xDenseC,
		yDenseC,
		zDenseC;

	/**
	 * Contents of the vectors
	 */
	protected double[] xdR, ydR, zdR, xdC, ydC, zdC;

	/**
	 * Tolerance for floating-point comparisons
	 */
	protected double tol = 1e-5;

	/**
	 * Maximum matrix size, to avoid too slow tests
	 */
	protected int max = 100;

	/**
	 * Constructor for MatrixTest
	 */
	public MatrixTest(String arg0) {
		super(arg0);
	}

	protected void setUp() throws Exception {
		setup();
		createAuxillerary();
	}

	protected abstract void setup() throws Exception;

	protected void tearDown() throws Exception {
		A = B = Bdense = null;
		xC =
			xDenseC =
				xDenseR =
					xR =
						yC =
							yDenseC =
								yDenseR =
									yR = zC = zDenseC = zDenseR = zR = null;
		Ad = Bd = null;
		xdC = xdR = ydC = ydR = zdC = zdR;
	}

	/**
	 * Called after setUp() to create additional datastructures
	 */
	protected void createAuxillerary() {
		Bdense = Matrices.random(A.numRows(), A.numColumns());
		B = Matrices.synchronizedMatrix(Bdense.copy());
		Bd = Matrices.getArray(B);

		xDenseC = Matrices.random(A.numColumns());
		yDenseC = Matrices.random(A.numColumns());
		zDenseC = Matrices.random(A.numColumns());

		xDenseR = Matrices.random(A.numRows());
		yDenseR = Matrices.random(A.numRows());
		zDenseR = Matrices.random(A.numRows());

		xC = Matrices.synchronizedVector(xDenseC);
		yC = Matrices.synchronizedVector(yDenseC);
		zC = Matrices.synchronizedVector(zDenseC);

		xR = Matrices.synchronizedVector(xDenseR);
		yR = Matrices.synchronizedVector(yDenseR);
		zR = Matrices.synchronizedVector(zDenseR);

		xdC = Matrices.getArray(xC);
		ydC = Matrices.getArray(yC);
		zdC = Matrices.getArray(zC);

		xdR = Matrices.getArray(xR);
		ydR = Matrices.getArray(yR);
		zdR = Matrices.getArray(zR);
	}

	public void testMatrixRank2Dense() {
		if (A.isSquare()) {
			int n = Utilities.getInt(max);
			Matrix B = Matrices.random(A.numRows(), n),
				C = Matrices.random(A.numRows(), n);
			double[][] Bd = Matrices.getArray(B), Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.rank2(alpha, B, C, beta);
			rank2(Ad, alpha, Bd, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Bd, B);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixRank2() {
		if (A.isSquare()) {
			int n = Utilities.getInt(max);
			Matrix B =
				Matrices.synchronizedMatrix(Matrices.random(A.numRows(), n)),
				C =
					Matrices.synchronizedMatrix(
						Matrices.random(A.numRows(), n));
			double[][] Bd = Matrices.getArray(B), Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.rank2(alpha, B, C, beta);
			rank2(Ad, alpha, Bd, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Bd, B);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixTransRank2Dense() {
		if (A.isSquare()) {
			int n = Utilities.getInt(max);
			Matrix B = Matrices.random(n, A.numColumns()),
				C = Matrices.random(n, A.numColumns());
			double[][] Bd = Matrices.getArray(B), Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.transRank2(alpha, B, C, beta);
			transRank2(Ad, alpha, Bd, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Bd, B);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixTransRank2() {
		if (A.isSquare()) {
			int n = Utilities.getInt(max);
			Matrix B =
				Matrices.synchronizedMatrix(Matrices.random(n, A.numColumns())),
				C =
					Matrices.synchronizedMatrix(
						Matrices.random(n, A.numColumns()));
			double[][] Bd = Matrices.getArray(B), Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.transRank2(alpha, B, C, beta);
			transRank2(Ad, alpha, Bd, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Bd, B);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixRank1Dense() {
		if (A.isSquare()) {
			Matrix C = Matrices.random(A.numRows(), A.numColumns());
			double[][] Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.rank1(alpha, C, beta);
			rank1(Ad, alpha, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixRank1() {
		if (A.isSquare()) {
			Matrix C =
				Matrices.synchronizedMatrix(
					Matrices.random(A.numRows(), A.numColumns()));
			double[][] Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.rank1(alpha, C, beta);
			rank1(Ad, alpha, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixTransRank1Dense() {
		if (A.isSquare()) {
			Matrix C = Matrices.random(A.numRows(), A.numColumns());
			double[][] Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.transRank1(alpha, C, beta);
			transRank1(Ad, alpha, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixTransRank1() {
		if (A.isSquare()) {
			Matrix C =
				Matrices.synchronizedMatrix(
					Matrices.random(A.numRows(), A.numColumns()));
			double[][] Cd = Matrices.getArray(C);
			double alpha = Math.random(), beta = Math.random();

			A = A.transRank1(alpha, C, beta);
			transRank1(Ad, alpha, Cd, beta);

			assertEquals(Ad, A);
			assertEquals(Cd, C);
		}
	}

	public void testMatrixMultAddDense() {
		int m = A.numRows(), k = A.numColumns(), n = Utilities.getInt(max);
		Matrix B = Matrices.random(k, n),
			C = Matrices.random(m, n),
			D = Matrices.random(m, n);
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.multAdd(alpha, B, beta, C, D);
		Dd = multAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixMultAdd() {
		int m = A.numRows(), k = A.numColumns(), n = Utilities.getInt(max);
		Matrix B = Matrices.synchronizedMatrix(Matrices.random(k, n)),
			C = Matrices.synchronizedMatrix(Matrices.random(m, n)),
			D = Matrices.synchronizedMatrix(Matrices.random(m, n));
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.multAdd(alpha, B, beta, C, D);
		Dd = multAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixTransAmultAddDense() {
		int m = A.numColumns(), k = A.numRows(), n = Utilities.getInt(max);
		Matrix B = Matrices.random(k, n),
			C = Matrices.random(m, n),
			D = Matrices.random(m, n);
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.transAmultAdd(alpha, B, beta, C, D);
		Dd = transAmultAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixTransAmultAdd() {
		int m = A.numColumns(), k = A.numRows(), n = Utilities.getInt(max);
		Matrix B = Matrices.synchronizedMatrix(Matrices.random(k, n)),
			C = Matrices.synchronizedMatrix(Matrices.random(m, n)),
			D = Matrices.synchronizedMatrix(Matrices.random(m, n));
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.transAmultAdd(alpha, B, beta, C, D);
		Dd = transAmultAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixTransABmultAddDense() {
		int m = A.numColumns(), k = A.numRows(), n = Utilities.getInt(max);
		Matrix B = Matrices.random(n, k),
			C = Matrices.random(m, n),
			D = Matrices.random(m, n);
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.transABmultAdd(alpha, B, beta, C, D);
		Dd = transABmultAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixTransABmultAdd() {
		int m = A.numColumns(), k = A.numRows(), n = Utilities.getInt(max);
		Matrix B = Matrices.synchronizedMatrix(Matrices.random(n, k)),
			C = Matrices.synchronizedMatrix(Matrices.random(m, n)),
			D = Matrices.synchronizedMatrix(Matrices.random(m, n));
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.transABmultAdd(alpha, B, beta, C, D);
		Dd = transABmultAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixTransBmultAddDense() {
		int m = A.numRows(), k = A.numColumns(), n = Utilities.getInt(max);
		Matrix B = Matrices.random(n, k),
			C = Matrices.random(m, n),
			D = Matrices.random(m, n);
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.transBmultAdd(alpha, B, beta, C, D);
		Dd = transBmultAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	public void testMatrixTransBmultAdd() {
		int m = A.numRows(), k = A.numColumns(), n = Utilities.getInt(max);
		Matrix B = Matrices.synchronizedMatrix(Matrices.random(n, k)),
			C = Matrices.synchronizedMatrix(Matrices.random(m, n)),
			D = Matrices.synchronizedMatrix(Matrices.random(m, n));
		double[][] Bd = Matrices.getArray(B),
			Cd = Matrices.getArray(C),
			Dd = Matrices.getArray(D);
		double alpha = Math.random(), beta = Math.random();

		D = A.transBmultAdd(alpha, B, beta, C, D);
		Dd = transBmultAdd(Ad, alpha, Bd, beta, Cd, Dd);

		assertEquals(Dd, D);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
		assertEquals(Cd, C);
	}

	protected double[][] rank2(
		double[][] Ad,
		double alpha,
		double[][] Bd,
		double[][] Cd,
		double beta) {
		return transBmultAdd(
			Bd,
			alpha,
			Cd,
			1,
			transBmultAdd(Cd, alpha, Bd, beta, Ad, Ad),
			Ad);
	}

	protected double[][] transRank2(
		double[][] Ad,
		double alpha,
		double[][] Bd,
		double[][] Cd,
		double beta) {
		return transAmultAdd(
			Bd,
			alpha,
			Cd,
			1,
			transAmultAdd(Cd, alpha, Bd, beta, Ad, Ad),
			Ad);
	}

	protected double[][] rank1(
		double[][] Ad,
		double alpha,
		double[][] Cd,
		double beta) {
		return transBmultAdd(Cd, alpha, Cd, beta, Ad, Ad);
	}

	protected double[][] transRank1(
		double[][] Ad,
		double alpha,
		double[][] Cd,
		double beta) {
		return transAmultAdd(Cd, alpha, Cd, beta, Ad, Ad);
	}

	public void testRank2doubleVectorVectorDense() {
		if (A.isSquare()) {
			double alpha = Math.random();
			assertEquals(
				rank2(alpha, xdR, ydR),
				A.rank2(alpha, xDenseR, yDenseR));
		}
	}

	public void testRank2doubleVectorVector() {
		if (A.isSquare()) {
			double alpha = Math.random();
			assertEquals(rank2(alpha, xdR, ydR), A.rank2(alpha, xR, yR));
		}
	}

	public void testRank1doubleVectorVectorDense() {
		if (A.isSquare()) {
			double alpha = Math.random();
			assertEquals(
				rank1(alpha, xdR, ydR),
				A.rank1(alpha, xDenseR, yDenseR));
		}
	}

	public void testRank1doubleVectorVector() {
		if (A.isSquare()) {
			double alpha = Math.random();
			assertEquals(rank1(alpha, xdR, ydR), A.rank1(alpha, xR, yR));
		}
	}

	protected double[][] rank2(double alpha, double[] xd, double[] yd) {
		rank1(alpha, xd, yd);
		rank1(alpha, yd, xd);
		return Ad;
	}

	protected double[][] rank1(double alpha, double[] xd, double[] yd) {
		for (int i = 0; i < xd.length; ++i)
			for (int j = 0; j < yd.length; ++j)
				Ad[i][j] += alpha * xd[i] * yd[j];
		return Ad;
	}

	public void testTransMultAdddoubleVectordoubleVectorVectorDense() {
		double alpha = Math.random(), beta = Math.random();
		assertEquals(
			transMultAdd(alpha, xdR, beta, ydC, zdC),
			A.transMultAdd(alpha, xDenseR, beta, yDenseC, zDenseC));
		assertEquals(Ad, A);
		assertEquals(xdR, xDenseR);
		assertEquals(ydC, yDenseC);
		assertEquals(zdC, zDenseC);
	}

	public void testTransMultAdddoubleVectordoubleVectorVector() {
		double alpha = Math.random(), beta = Math.random();
		assertEquals(
			transMultAdd(alpha, xdR, beta, ydC, zdC),
			A.transMultAdd(alpha, xR, beta, yC, zC));
		assertEquals(Ad, A);
		assertEquals(xdR, xR);
		assertEquals(ydC, yC);
		assertEquals(zdC, zC);
	}

	protected double[] transMultAdd(
		double alpha,
		double[] xd,
		double beta,
		double[] yd,
		double[] zd) {
		for (int i = 0; i < zd.length; ++i)
			zd[i] = beta * yd[i];

		int rows = Ad.length, cols = 0;
		if (rows > 0)
			cols = Ad[0].length;

		for (int j = 0; j < cols; ++j) {
			double dot = 0;
			for (int i = 0; i < rows; ++i)
				dot += Ad[i][j] * xd[i];
			zd[j] += alpha * dot;
		}

		return zd;
	}

	public void testMultAdddoubleVectordoubleVectorVectorDense() {
		double alpha = Math.random(), beta = Math.random();
		assertEquals(
			multAdd(alpha, xdC, beta, ydR, zdR),
			A.multAdd(alpha, xDenseC, beta, yDenseR, zDenseR));
		assertEquals(Ad, A);
		assertEquals(xdC, xDenseC);
		assertEquals(ydR, yDenseR);
		assertEquals(zdR, zDenseR);
	}

	public void testMultAdddoubleVectordoubleVectorVector() {
		double alpha = Math.random(), beta = Math.random();
		assertEquals(
			multAdd(alpha, xdC, beta, ydR, zdR),
			A.multAdd(alpha, xC, beta, yR, zR));
		assertEquals(Ad, A);
		assertEquals(xdC, xC);
		assertEquals(ydR, yR);
		assertEquals(zdR, zR);
	}

	protected double[] multAdd(
		double alpha,
		double[] xd,
		double beta,
		double[] yd,
		double[] zd) {
		for (int i = 0; i < Ad.length; ++i) {
			double dot = 0;
			for (int j = 0; j < Ad[i].length; ++j)
				dot += Ad[i][j] * xd[j];
			zd[i] = alpha * dot + beta * yd[i];
		}
		return zd;
	}

	protected double[][] multAdd(
		double[][] Ad,
		double alpha,
		double[][] Bd,
		double beta,
		double[][] Cd,
		double[][] Dd) {
		int m = Dd.length, n = 0, k = Bd.length;
		if (k > 0)
			n = Bd[0].length;

		for (int j = 0; j < n; ++j) {
			for (int i = 0; i < m; ++i)
				Dd[i][j] = beta * Cd[i][j];
			for (int l = 0; l < k; ++l)
				for (int i = 0; i < m; ++i)
					Dd[i][j] += alpha * Ad[i][l] * Bd[l][j];
		}

		return Dd;
	}

	protected double[][] transAmultAdd(
		double[][] Ad,
		double alpha,
		double[][] Bd,
		double beta,
		double[][] Cd,
		double[][] Dd) {
		int m = Dd.length, n = 0, k = Bd.length;
		if (k > 0)
			n = Bd[0].length;

		for (int j = 0; j < n; ++j)
			for (int i = 0; i < m; ++i) {
				double temp = 0;
				for (int l = 0; l < k; ++l)
					temp += Ad[l][i] * Bd[l][j];
				Dd[i][j] = alpha * temp + beta * Cd[i][j];
			}

		return Dd;
	}

	protected double[][] transBmultAdd(
		double[][] Ad,
		double alpha,
		double[][] Bd,
		double beta,
		double[][] Cd,
		double[][] Dd) {
		int m = Dd.length, n = Bd.length, k = 0;
		if (n > 0)
			k = Bd[0].length;

		for (int j = 0; j < n; ++j) {
			for (int i = 0; i < m; ++i)
				Dd[i][j] = beta * Cd[i][j];
			for (int l = 0; l < k; ++l)
				for (int i = 0; i < m; ++i)
					Dd[i][j] += alpha * Ad[i][l] * Bd[j][l];
		}

		return Dd;
	}

	protected double[][] transABmultAdd(
		double[][] Ad,
		double alpha,
		double[][] Bd,
		double beta,
		double[][] Cd,
		double[][] Dd) {
		int m = Dd.length, n = Bd.length, k = 0;
		if (n > 0)
			k = Bd[0].length;

		for (int j = 0; j < n; ++j)
			for (int i = 0; i < m; ++i) {
				double temp = 0;
				for (int l = 0; l < k; ++l)
					temp += Ad[l][i] * Bd[j][l];
				Dd[i][j] = alpha * temp + beta * Cd[i][j];
			}

		return Dd;
	}

	/**
	 * Tests <code>A = A + alpha*B</code>
	 */
	public void testRandomMatrixAdd() {
		double alpha = Math.random();
		A = A.add(alpha, B);
		add(Ad, alpha, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = A + B</code>
	 */
	public void testMatrixAdd() {
		A = A.add(B);
		add(Ad, 1, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = A + 1*B</code>
	 */
	public void testOneMatrixAdd() {
		A = A.add(1, B);
		add(Ad, 1, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = A + 0*B</code>
	 */
	public void testZeroMatrixAdd() {
		A = A.add(0, B);
		add(Ad, 0, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = alpha*B</code>
	 */
	public void testRandomMatrixSet() {
		double alpha = Math.random();
		A = A.set(alpha, B);
		set(Ad, alpha, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = B</code>
	 */
	public void testMatrixSet() {
		A = A.set(B);
		set(Ad, 1, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = 1*B</code>
	 */
	public void testOneMatrixSet() {
		A = A.set(1, B);
		set(Ad, 1, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Tests <code>A = 0*B</code>
	 */
	public void testZeroMatrixSet() {
		A = A.set(0, B);
		set(Ad, 0, Bd);
		assertEquals(Ad, A);
		assertEquals(Bd, B);
	}

	/**
	 * Checks transpose
	 */
	public void testTranspose() {
		Matrix At = Matrices.random(A.numColumns(), A.numRows());
		assertEquals(transpose(), A.transpose(At));
	}

	protected void set(double[][] A, double alpha, double[][] B) {
		for (int i = 0; i < A.length; ++i)
			for (int j = 0; j < A[i].length; ++j)
				A[i][j] = alpha * B[i][j];
	}

	protected void add(double[][] A, double alpha, double[][] B) {
		for (int i = 0; i < A.length; ++i)
			for (int j = 0; j < A[i].length; ++j)
				A[i][j] += alpha * B[i][j];
	}

	protected double[][] transpose() {
		if (Ad.length == 0)
			return new double[0][0];

		double[][] Adt = new double[Ad[0].length][Ad.length];
		for (int i = 0; i < Ad.length; ++i)
			for (int j = 0; j < Ad[i].length; ++j)
				Adt[j][i] = Ad[i][j];
		return Adt;
	}

	/**
	 * Test of direct matrix solver
	 */
	public void testMatrixSolve() {
		while (true) {
			try {
				Matrix B = Matrices.random(A.numRows(), A.numColumns()),
					X = Matrices.random(A.numRows(), A.numColumns());
				X = A.solve(B, X);

				Matrix Y = A.multAdd(X, -1, B, X.copy().zero());
				assertEquals(0, Y.norm(Matrix.Norm.Frobenius), tol);
				assertEquals(Ad, A);
				return;
			} catch (MatrixSingularException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			} catch (MatrixNotSPDException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			}
		}
	}

	/**
	 * Test of direct transpose matrix solver
	 */
	public void testTransMatrixSolve() {
		while (true) {
			try {
				Matrix B = Matrices.random(A.numRows(), A.numColumns()),
					X = Matrices.random(A.numRows(), A.numColumns());
				X = A.transSolve(B, X);

				Matrix Y = A.transAmultAdd(X, -1, B, X.copy().zero());
				assertEquals(0, Y.norm(Matrix.Norm.Frobenius), tol);
				assertEquals(Ad, A);
				return;
			} catch (MatrixSingularException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			} catch (MatrixNotSPDException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			}
		}
	}

	/**
	 * Test of direct vector solver
	 */
	public void testVectorSolve() {
		while (true) {
			try {
				Vector b = Matrices.random(A.numRows()),
					x = Matrices.random(A.numRows());
				x = A.solve(b, x);

				Vector y = A.multAdd(x, -1, b, x.copy().zero());
				assertEquals(0, y.norm(Vector.Norm.Two), tol);
				assertEquals(Ad, A);
				return;
			} catch (MatrixSingularException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			} catch (MatrixNotSPDException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			}
		}
	}

	/**
	 * Test of direct transpose vector solver
	 */
	public void testTransVectorSolve() {
		while (true) {
			try {
				Vector b = Matrices.random(A.numRows()),
					x = Matrices.random(A.numRows());
				x = A.transSolve(b, x);

				Vector y = A.transMultAdd(x, -1, b, x.copy().zero());
				assertEquals(0, y.norm(Vector.Norm.Two), tol);
				assertEquals(Ad, A);
				return;
			} catch (MatrixSingularException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			} catch (MatrixNotSPDException e) {
				A.addDiagonal(1);
				addDiagonal(1);
			}
		}
	}

	/**
	 * Test additions using iterators
	 */
	public void testAdd() {
		double alpha = Math.random();
		/*for (MatrixEntry e : A) {
			A.add(e.row(), e.column(), alpha);
			A.add(e.row(), e.column(), -alpha);
		}*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			A.add(e.row(), e.column(), alpha);
			A.add(e.row(), e.column(), -alpha);
		}
		assertEquals(Ad, A);
	}

	/**
	 * Checks that copy is deep, not reference
	 */
	public void testCopy() {
		Matrix Ac = A.copy();
		A = A.zero();
		assertEquals(Ad, Ac);
	}

	/**
	 * Test iterator get
	 */
	public void testIterator() {
		double[][] Ac = new double[A.numRows()][A.numColumns()];
		//for (MatrixEntry e : A)
			//Ac[e.row()][e.column()] = e.get();
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			Ac[e.row()][e.column()] = e.get();
		}
		assertEquals(Ad, Ac);
	}

	/**
	 * Test iterator set
	 */
	public void testIteratorSet() {
		double alpha = Math.random();
		/*for (MatrixEntry e : A)
			e.set(e.get() * alpha);*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			e.set(e.get() * alpha);
		}
		assertEquals(scale(alpha), A);
	}

	/**
	 * Test iterator read and write
	 */
	public void testIteratorSetGet() {
		double alpha = Math.random();
		double[][] Ac = new double[A.numRows()][A.numColumns()];
		/*for (MatrixEntry e : A) {
			Ac[e.row()][e.column()] = e.get();
			e.set(alpha * e.get());
			e.set(e.get() / alpha);
		}*/
		MatrixEntry e;
		Iterator iter = A.iterator();
		while(iter.hasNext()) {
			e = (MatrixEntry) iter.next();
			Ac[e.row()][e.column()] = e.get();
			e.set(alpha * e.get());
			e.set(e.get() / alpha);
		}
		assertEquals(Ad, Ac);
		assertEquals(Ad, A);
	}

	/**
	 * Checks zero()
	 */
	public void testZero() {
		assertEquals(zero(), A.zero());
	}

	protected double[][] zero() {
		for (int i = 0; i < Ad.length; ++i)
			for (int j = 0; j < Ad[i].length; ++j)
				Ad[i][j] = 0;
		return Ad;
	}

	/**
	 * Adds a random scalar to the main diagonal
	 */
	public void testAddDiagonal() {
		double alpha = Math.random();
		assertEquals(addDiagonal(alpha), A.addDiagonal(alpha));
	}

	/**
	 * Adds one to the main diagonal
	 */
	public void testAddOneDiagonal() {
		assertEquals(addDiagonal(1), A.addDiagonal(1));
	}

	/**
	 * Adds zero to the main diagonal
	 */
	public void testAddZeroDiagonal() {
		assertEquals(addDiagonal(0), A.addDiagonal(0));
	}

	protected double[][] addDiagonal(double value) {
		int max = Ad.length;
		if (max > 0)
			max = Math.min(max, Ad[0].length);

		for (int i = 0; i < max; ++i)
			Ad[i][i] += value;
		return Ad;
	}

	/**
	 * Cardinality computation
	 */
	public void testCardinality() {
		assertEquals(A.cardinality(), cardinality());
	}

	protected int cardinality() {
		int nz = 0;
		for (int i = 0; i < Ad.length; ++i)
			for (int j = 0; j < Ad[i].length; ++j)
				if (Ad[i][j] != 0.)
					nz++;
		return nz;
	}

	/**
	 * Checks in-place transpose for square matrices
	 */
	public void testTransposeInplace() {
		if (A.isSquare())
			assertEquals(transpose(), A.copy().transpose());
	}

	/**
	 * Scaling with an arbitrary scalar
	 */
	public void testScale() {
		double alpha = Math.random();
		A = A.scale(alpha);
		scale(alpha);
		assertEquals(Ad, A);
	}

	/**
	 * Scaling by zero
	 */
	public void testZeroScale() {
		A = A.scale(0);
		scale(0);
		assertEquals(Ad, A);
	}

	/**
	 * Scaling by one
	 */
	public void testOneScale() {
		A = A.scale(1);
		scale(1);
		assertEquals(Ad, A);
	}

	protected double[][] scale(double alpha) {
		for (int i = 0; i < Ad.length; ++i)
			for (int j = 0; j < Ad[i].length; ++j)
				Ad[i][j] *= alpha;
		return Ad;
	}

	/**
	 * Checks the 1 norm
	 */
	public void testOneNorm() {
		assertEquals(norm1(Ad), A.norm(Matrix.Norm.One), tol);
		assertEquals(Ad, A);
	}

	/**
	 * Checks the Frobenius norm
	 */
	public void testFrobeniusNorm() {
		assertEquals(normF(Ad), A.norm(Matrix.Norm.Frobenius), tol);
		assertEquals(Ad, A);
	}

	/**
	 * Checks the infinity norm
	 */
	public void testInfinityNorm() {
		assertEquals(normInf(Ad), A.norm(Matrix.Norm.Infinity), tol);
		assertEquals(Ad, A);
	}

	protected double norm1(double[][] A) {
		double max = 0;
		for (int i = 0; i < A.length; ++i) {
			double rowsum = 0;
			for (int j = 0; j < A[i].length; ++j)
				rowsum += Math.abs(A[i][j]);
			max = Math.max(rowsum, max);
		}
		return max;
	}

	protected double normF(double[][] A) {
		double norm = 0;
		for (int i = 0; i < A.length; ++i)
			for (int j = 0; j < A[i].length; ++j)
				norm += A[i][j] * A[i][j];
		return Math.sqrt(norm);
	}

	protected double normInf(double[][] A) {
		if (A.length == 0)
			return 0;

		double[] columnSum = new double[A[0].length];
		for (int i = 0; i < A.length; ++i)
			for (int j = 0; j < A[i].length; ++j)
				columnSum[j] += Math.abs(A[i][j]);

		double max = 0;
		/*for (double d : columnSum)
			max = Math.max(max, d);*/
		for(int i = 0; i < columnSum.length; ++i) {
			max = Math.max(max, columnSum[i]);
		}
		return max;
	}

	/**
	 * Checks for equality between the matrix and the array
	 */
	protected void assertEquals(double[][] Ad, Matrix A) {
		assertTrue(A != null);
		assertTrue(Ad != null);
		assertTrue(A.numRows() == Ad.length);
		for (int i = 0; i < A.numRows(); ++i) {
			assertTrue(A.numColumns() == Ad[i].length);
			for (int j = 0; j < A.numColumns(); ++j)
				assertEquals(Ad[i][j], A.get(i, j), 1e-12);
		}
	}

	/**
	 * Checks for equality between two arrays
	 */
	protected void assertEquals(double[][] Ad, double[][] Ac) {
		assertTrue(Ac.length == Ad.length);
		for (int i = 0; i < A.numRows(); ++i) {
			assertTrue(Ac[i].length == Ad[i].length);
			for (int j = 0; j < A.numColumns(); ++j)
				assertEquals(Ad[i][j], Ac[i][j], 1e-12);
		}
	}

	protected void assertEquals(double[] xd, Vector x) {
		assertEquals(xd.length, x.size());
		for (int i = 0; i < xd.length; ++i)
			assertEquals(xd[i], x.get(i), tol);
	}

	protected void assertEquals(double[] xd, double[] yd) {
		for (int i = 0; i < xd.length; ++i)
			assertEquals(xd[i], yd[i], tol);
	}

}
