/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import mt.ll.Interface;
import mt.ll.BLASkernel.Diag;
import mt.ll.BLASkernel.Transpose;
import mt.ll.BLASkernel.UpLo;

/**
 * Partial implementation of a triangular, banded matrix
 */
abstract class AbstractTriangBandMatrix extends AbstractBandMatrix {

	/**
	 * Upper or lower part stored?
	 */
	private UpLo uplo;

	/**
	 * Diagonal stored or not?
	 */
	private Diag diag;

	/**
	 * Diagonals in relevant band
	 */
	int kd;

	/**
	 * Constructor for AbstractTriangBandMatrix
	 */
	AbstractTriangBandMatrix(int n, int kl, int ku, UpLo uplo, Diag diag) {
		super(n, kl, ku);
		kd = Math.max(kl, ku);
		this.uplo = uplo;
		this.diag = diag;
	}

	/**
	 * Constructor for AbstractTriangBandMatrix
	 */
	AbstractTriangBandMatrix(Matrix A, int kl, int ku, UpLo uplo, Diag diag) {
		this(A, kl, ku, true, uplo, diag);
	}

	/**
	 * Constructor for AbstractTriangBandMatrix
	 */
	AbstractTriangBandMatrix(
		Matrix A,
		int kl,
		int ku,
		boolean deep,
		UpLo uplo,
		Diag diag) {
		super(A, kl, ku, deep);
		kd = Math.max(kl, ku);
		this.uplo = uplo;
		this.diag = diag;
	}

	public Vector multAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(z instanceof DenseVector))
			return super.multAdd(alpha, x, beta, y, z);

		checkMultAdd(x, y, z);

		double[] zd = ((DenseVector) z).getData();

		// z = alpha*x
		z.set(alpha, x);

		// z = A*z
		Interface.blas().tbmv(
			uplo,
			Transpose.NoTranspose,
			diag,
			numRows,
			kd,
			data,
			kd + 1,
			zd);

		// z = z + beta*y
		return z.add(beta, y);
	}

	public Vector transMultAdd(
		double alpha,
		Vector x,
		double beta,
		Vector y,
		Vector z) {
		if (!(z instanceof DenseVector))
			return super.transMultAdd(alpha, x, beta, y, z);

		checkTransMultAdd(x, y, z);

		double[] zd = ((DenseVector) z).getData();

		// z = alpha*x
		z.set(alpha, x);

		// z = A*z
		Interface.blas().tbmv(
			uplo,
			Transpose.Transpose,
			diag,
			numRows,
			kd,
			data,
			kd + 1,
			zd);

		// z = z + beta*y
		return z.add(beta, y);
	}

	public Matrix solve(Matrix B, Matrix X) {
		return solve(B, X, Transpose.NoTranspose);
	}

	public Vector solve(Vector b, Vector x) {
		DenseMatrix B = new DenseMatrix(b, false),
			X = new DenseMatrix(x, false);
		solve(B, X);
		return x;
	}

	public Matrix transSolve(Matrix B, Matrix X) {
		return solve(B, X, Transpose.Transpose);
	}

	public Vector transSolve(Vector b, Vector x) {
		DenseMatrix B = new DenseMatrix(b, false),
			X = new DenseMatrix(x, false);
		transSolve(B, X);
		return x;
	}

	Matrix solve(Matrix B, Matrix X, Transpose trans) {
		if (!(X instanceof DenseMatrix))
			throw new UnsupportedOperationException("X must be a DenseMatrix");

		checkSolve(B, X);

		double[] Xd = ((DenseMatrix) X).getData();

		X.set(B);

		int info =
			Interface.lapack().tbtrs(
				uplo,
				trans,
				diag,
				numRows,
				kd,
				X.numColumns(),
				data,
				Xd);

		if (info > 0)
			throw new MatrixSingularException();
		else if (info < 0)
			throw new IllegalArgumentException();

		return X;
	}

}
