/*
 * Decompiled with CFR 0.152.
 */
package jannovar.pedigree;

import jannovar.common.Disease;
import jannovar.common.Genotype;
import jannovar.exception.PedParseException;
import jannovar.exome.Variant;
import jannovar.genotype.GenotypeCall;
import jannovar.pedigree.Person;
import java.util.ArrayList;
import java.util.HashMap;

public class Pedigree {
    private String familyID = null;
    private String singleSampleName = null;
    private ArrayList<Person> personList = null;
    private ArrayList<Person> affectedList = null;
    private ArrayList<Person> parentList = null;
    private ArrayList<Person> unaffectedList = null;
    private boolean isSingleSample;

    public static Pedigree constructSingleSamplePedigree(String name) {
        Pedigree ped = new Pedigree();
        ped.singleSampleName = name;
        ped.isSingleSample = true;
        return ped;
    }

    private Pedigree() {
    }

    public Pedigree(ArrayList<Person> pList, String famID) throws PedParseException {
        this.personList = new ArrayList();
        this.familyID = famID;
        for (Person p : pList) {
            this.addIndividual(p);
        }
        boolean success = this.findParentLinks();
        if (!success) {
            throw new PedParseException("Inconsistent Parent Relations in PED file");
        }
        this.setPersonIndices();
        this.initializeAffectedsParentsSibs();
        this.isSingleSample = this.personList.size() == 1;
    }

    public int getPedigreeSize() {
        return this.personList.size();
    }

    public String getSingleSampleName() {
        if (this.isSingleSample) {
            return this.singleSampleName;
        }
        return "[Pedigree] Error: Attempt to retrieve single sampl name for multiple sample PED file";
    }

    public boolean isNthPersonAffected(int n) {
        if (n < 0 || n >= this.personList.size()) {
            return false;
        }
        Person p = this.personList.get(n);
        return p.isAffected();
    }

    public ArrayList<String> getPEDFileDatForNthPerson(int n) {
        if (n < 0 || n >= this.personList.size()) {
            return null;
        }
        Person p = this.personList.get(n);
        return p.getPEDFileData();
    }

    public boolean isNthPersonParentOfAffected(int n) {
        if (n < 0 || n >= this.personList.size()) {
            return false;
        }
        Person p = this.personList.get(n);
        return this.parentList.contains(p);
    }

    public boolean sampleIsRepresentedInPedigree(String name) {
        for (Person p : this.personList) {
            if (!name.equals(p.getIndividualID())) continue;
            return true;
        }
        return false;
    }

    public void adjustSampleOrderInPedFile(ArrayList<String> sampleNames) throws PedParseException {
        StringBuilder sb;
        if (sampleNames == null) {
            String e = "[Pedigree:adjustSampleOrderInPedFile] Error: VCF sample name list empty";
            throw new PedParseException(e);
        }
        if (sampleNames.size() != this.getPedigreeSize()) {
            String e = String.format("[Pedigree:adjustSampleOrderInPedFile] Error:%n individuals in pedigree but %d individuals in the VCF file", sampleNames.size(), this.getPedigreeSize());
            throw new PedParseException(e);
        }
        ArrayList<String> badName = new ArrayList<String>();
        for (String s : sampleNames) {
            if (this.sampleIsRepresentedInPedigree(s)) continue;
            badName.add(s);
        }
        if (badName.size() > 0) {
            sb = new StringBuilder();
            sb.append("[Pedigree:adjustSampleOrderInPedFile] Error: Did not find VCF sample names in PED file: ");
            boolean first = true;
            for (String s : badName) {
                if (first) {
                    sb.append(s);
                    first = false;
                    continue;
                }
                sb.append(", ").append(s);
            }
            throw new PedParseException(sb.toString());
        }
        for (Person p : this.personList) {
            String id = p.getIndividualID();
            if (sampleNames.contains(id)) continue;
            badName.add(id);
        }
        if (badName.size() > 0) {
            sb = new StringBuilder();
            sb.append("[Pedigree:adjustSampleOrderInPedFile] Error: Did not find PED file names in VCF file: ");
            boolean first = true;
            for (String s : badName) {
                if (first) {
                    sb.append(s);
                    first = false;
                    continue;
                }
                sb.append(", ").append(s);
            }
            throw new PedParseException(sb.toString());
        }
        ArrayList<Person> newList = new ArrayList<Person>();
        block4: for (String s : sampleNames) {
            for (Person p : this.personList) {
                if (!s.equals(p.getIndividualID())) continue;
                newList.add(p);
                continue block4;
            }
        }
        if (newList.size() != this.personList.size()) {
            String e = String.format("Error adjusting sample order.Added %d samples to new list but  the original PED file has %d", newList.size(), this.personList.size());
            throw new PedParseException(e);
        }
        this.personList = newList;
        this.setPersonIndices();
    }

    public void addIndividual(Person person) throws PedParseException {
        if (!this.familyID.equals(person.getFamilyID())) {
            String err = String.format("Attempt to add person with different family id (%s) to pedigree for family %s", person.getFamilyID(), this.familyID);
            throw new PedParseException(err);
        }
        this.personList.add(person);
    }

    public void setFamilyID(String id) {
        this.familyID = id;
    }

    private void setPersonIndices() {
        for (int i = 0; i < this.personList.size(); ++i) {
            Person p = this.personList.get(i);
            p.setIndex(i);
        }
    }

    private void initializeAffectedsParentsSibs() {
        Disease ds;
        this.affectedList = new ArrayList();
        this.parentList = new ArrayList();
        this.unaffectedList = new ArrayList();
        for (Person p : this.personList) {
            ds = p.getDiseaseStatus();
            if (ds != Disease.AFFECTED) continue;
            this.affectedList.add(p);
        }
        for (Person p : this.affectedList) {
            Person father = p.getFather();
            Person mother = p.getMother();
            if (father != null && !this.parentList.contains(father)) {
                this.parentList.add(father);
            }
            if (mother == null || this.parentList.contains(mother)) continue;
            this.parentList.add(mother);
        }
        for (Person p : this.personList) {
            ds = p.getDiseaseStatus();
            if (ds != Disease.UNAFFECTED || this.parentList.contains(p)) continue;
            this.unaffectedList.add(p);
        }
    }

    private boolean findParentLinks() throws PedParseException {
        HashMap<String, Person> personMap = new HashMap<String, Person>();
        for (Person p : this.personList) {
            String id = p.getIndividualID();
            personMap.put(id, p);
        }
        for (Person p : this.personList) {
            String fatherID = p.getFatherID();
            String motherID = p.getMotherID();
            if (fatherID != null) {
                Person father = (Person)personMap.get(fatherID);
                if (father == null) {
                    String s = String.format("[Pedigree] Could not find father id: %s", fatherID);
                    throw new PedParseException(s);
                }
                p.setFather(father);
            }
            if (motherID == null) continue;
            Person mother = (Person)personMap.get(motherID);
            if (mother == null) {
                String s = String.format("[Pedigree] Could not find mother id: %s", motherID);
                throw new PedParseException(s);
            }
            p.setMother(mother);
        }
        return true;
    }

    public int getNumberOfIndividualsInPedigree() {
        if (this.isSingleSample) {
            return 1;
        }
        return this.personList.size();
    }

    public int getNumberOfParentsInPedigree() {
        if (this.isSingleSample) {
            return 0;
        }
        return this.parentList.size();
    }

    public int getNumberOfAffectedsInPedigree() {
        if (this.isSingleSample) {
            return 1;
        }
        return this.affectedList.size();
    }

    public int getNumberOfUnaffectedsInPedigree() {
        if (this.isSingleSample) {
            return 0;
        }
        return this.unaffectedList.size();
    }

    public Person getPerson(String id) {
        for (Person p : this.personList) {
            if (!id.equals(p.getIndividualID())) continue;
            return p;
        }
        return null;
    }

    public boolean isCompatibleWithAutosomalDominant(ArrayList<Variant> varList) {
        if (this.isSingleSample) {
            return this.singleSampleHasHeterozygousVariant(varList);
        }
        for (Variant v : varList) {
            GenotypeCall multiGT = v.getGenotype();
            int N = multiGT.getNumberOfIndividuals();
            boolean variantCompatible = true;
            int n_affected_with_het = 0;
            for (int i = 0; i < N; ++i) {
                Genotype gt = multiGT.getGenotypeInIndividualN(i);
                Disease diseaseStatus = this.personList.get(i).getDiseaseStatus();
                if (diseaseStatus == Disease.AFFECTED) {
                    if (gt == Genotype.HOMOZYGOUS_REF || gt == Genotype.HOMOZYGOUS_ALT) {
                        variantCompatible = false;
                        break;
                    }
                    if (gt != Genotype.HETEROZYGOUS) continue;
                    variantCompatible = true;
                    ++n_affected_with_het;
                    continue;
                }
                if (diseaseStatus != Disease.UNAFFECTED || gt != Genotype.HETEROZYGOUS && gt != Genotype.HOMOZYGOUS_ALT) continue;
                variantCompatible = false;
                break;
            }
            if (!variantCompatible || n_affected_with_het <= 0) continue;
            return true;
        }
        return false;
    }

    private boolean affectedsAreHomozygousALT(GenotypeCall multiGT) {
        for (Person p : this.affectedList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (gt == Genotype.HOMOZYGOUS_ALT) continue;
            return false;
        }
        return true;
    }

    private boolean parentsAreHeterozygous(GenotypeCall multiGT) {
        for (Person p : this.parentList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (gt == Genotype.HETEROZYGOUS) continue;
            return false;
        }
        return true;
    }

    private boolean unaffectedsAreNotHomozygousALT(GenotypeCall multiGT) {
        for (Person p : this.unaffectedList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (gt != Genotype.HOMOZYGOUS_ALT) continue;
            return false;
        }
        return true;
    }

    private boolean affectedsAreHeterozygous(GenotypeCall multiGT) {
        for (Person p : this.affectedList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (gt == Genotype.HETEROZYGOUS) continue;
            return false;
        }
        return true;
    }

    private boolean onlyOneParentIsHeterozygous(GenotypeCall multiGT) {
        int n = 0;
        for (Person p : this.parentList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (gt != Genotype.HETEROZYGOUS) continue;
            ++n;
        }
        return n == 1;
    }

    private boolean fatherIsHeterozygous(GenotypeCall multiGT) {
        for (Person p : this.parentList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (!p.isMale() || gt != Genotype.HETEROZYGOUS) continue;
            return true;
        }
        return false;
    }

    private boolean motherIsHeterozygous(GenotypeCall multiGT) {
        for (Person p : this.parentList) {
            int idx = p.getIndex();
            Genotype gt = multiGT.getGenotypeInIndividualN(idx);
            if (!p.isFemale() || gt != Genotype.HETEROZYGOUS) continue;
            return true;
        }
        return false;
    }

    public boolean containsCompatibleHomozygousVariant(GenotypeCall GT) {
        if (this.isSingleSample) {
            return Genotype.HOMOZYGOUS_ALT == GT.getGenotypeInIndividualN(0);
        }
        return this.affectedsAreHomozygousALT(GT) && this.parentsAreHeterozygous(GT) && this.unaffectedsAreNotHomozygousALT(GT);
    }

    private boolean validCompoundHet(GenotypeCall matGT, GenotypeCall patGT) {
        for (Person p : this.unaffectedList) {
            int i = p.getIndex();
            Genotype g1 = matGT.getGenotypeInIndividualN(i);
            Genotype g2 = patGT.getGenotypeInIndividualN(i);
            if (g1 != Genotype.HETEROZYGOUS || g2 != Genotype.HETEROZYGOUS) continue;
            return false;
        }
        return true;
    }

    public boolean isCompatibleWithAutosomalRecessiveHomozygous(ArrayList<Variant> varList) {
        if (this.isSingleSample) {
            for (Variant v : varList) {
                GenotypeCall gc = v.getGenotype();
                Genotype g = gc.getGenotypeInIndividualN(0);
                if (g != Genotype.HOMOZYGOUS_ALT) continue;
                return true;
            }
            return false;
        }
        if (this.parentList.size() > 2) {
            throw new UnsupportedOperationException("Autosomal recessive pedigree analysis with more than two parents is not supported!");
        }
        for (Variant v : varList) {
            GenotypeCall multiGT = v.getGenotype();
            if (!this.containsCompatibleHomozygousVariant(multiGT)) continue;
            return true;
        }
        return false;
    }

    public boolean isCompatibleWithAutosomalRecessiveCompoundHet(ArrayList<Variant> varList) {
        if (this.isSingleSample) {
            int n_het = 0;
            for (Variant v : varList) {
                GenotypeCall gc = v.getGenotype();
                Genotype g = gc.getGenotypeInIndividualN(0);
                if (g != Genotype.HETEROZYGOUS) continue;
                ++n_het;
            }
            return n_het > 1;
        }
        ArrayList<GenotypeCall> paternal = new ArrayList<GenotypeCall>();
        ArrayList<GenotypeCall> maternal = new ArrayList<GenotypeCall>();
        if (this.parentList.size() > 2) {
            throw new UnsupportedOperationException("Autosomal recessive pedigree analysis with more than two parents is not supported!");
        }
        for (Variant v : varList) {
            GenotypeCall multiGT = v.getGenotype();
            if (!this.affectedsAreHeterozygous(multiGT) || !this.onlyOneParentIsHeterozygous(multiGT) || !this.unaffectedsAreNotHomozygousALT(multiGT)) continue;
            if (this.fatherIsHeterozygous(multiGT)) {
                paternal.add(multiGT);
                continue;
            }
            if (this.motherIsHeterozygous(multiGT)) {
                maternal.add(multiGT);
                continue;
            }
            System.err.println("ERROR: Neither mother nor father het with at least one parent being het");
            System.exit(1);
        }
        for (GenotypeCall patGT : paternal) {
            for (GenotypeCall matGT : maternal) {
                if (!this.validCompoundHet(matGT, patGT)) continue;
                return true;
            }
        }
        return false;
    }

    public boolean isCompatibleWithAutosomalRecessive(ArrayList<Variant> varList) {
        if (this.isSingleSample) {
            return this.singleSampleCompatibleWithAutosomalRecessive(varList);
        }
        ArrayList<GenotypeCall> paternal = new ArrayList<GenotypeCall>();
        ArrayList<GenotypeCall> maternal = new ArrayList<GenotypeCall>();
        if (this.parentList.size() > 2) {
            throw new UnsupportedOperationException("Autosomal recessive pedigree analysis with more than two parents is not supported!");
        }
        for (Variant v : varList) {
            GenotypeCall multiGT = v.getGenotype();
            if (this.containsCompatibleHomozygousVariant(multiGT)) {
                return true;
            }
            if (!this.affectedsAreHeterozygous(multiGT) || !this.onlyOneParentIsHeterozygous(multiGT) || !this.unaffectedsAreNotHomozygousALT(multiGT)) continue;
            if (this.fatherIsHeterozygous(multiGT)) {
                paternal.add(multiGT);
                continue;
            }
            if (this.motherIsHeterozygous(multiGT)) {
                maternal.add(multiGT);
                continue;
            }
            System.err.println("ERROR: Neither mother nor father het with at least one parent being het");
            System.exit(1);
        }
        for (GenotypeCall patGT : paternal) {
            for (GenotypeCall matGT : maternal) {
                if (!this.validCompoundHet(matGT, patGT)) continue;
                return true;
            }
        }
        return false;
    }

    public boolean isCompatibleWithXChromosomalRecessive(ArrayList<Variant> varList) {
        if (varList.isEmpty()) {
            System.out.println("[Pedigree.java] Warning: attempt to test zero-length variant list");
            return false;
        }
        if (!varList.get(0).is_X_chromosomal()) {
            return false;
        }
        if (this.isSingleSample) {
            for (Variant v : varList) {
                GenotypeCall gc = v.getGenotype();
                Genotype g = gc.getGenotypeInIndividualN(0);
                if (g != Genotype.HOMOZYGOUS_ALT) continue;
                return true;
            }
            return false;
        }
        for (Variant v : varList) {
            Genotype g;
            int i;
            GenotypeCall gc = v.getGenotype();
            boolean compatible = true;
            for (Person p : this.affectedList) {
                i = p.getIndex();
                g = gc.getGenotypeInIndividualN(i);
                if (g != Genotype.HOMOZYGOUS_ALT) {
                    compatible = false;
                    break;
                }
                if (compatible) continue;
                break;
            }
            for (Person p : this.parentList) {
                i = p.getIndex();
                g = gc.getGenotypeInIndividualN(i);
                if (p.isMale() && !p.isAffected() && g == Genotype.HOMOZYGOUS_ALT) {
                    compatible = false;
                    break;
                }
                if (p.isFemale() && g != Genotype.HETEROZYGOUS) {
                    compatible = false;
                    break;
                }
                if (compatible) continue;
                break;
            }
            for (Person p : this.unaffectedList) {
                i = p.getIndex();
                g = gc.getGenotypeInIndividualN(i);
                if (p.isMale() && g == Genotype.HOMOZYGOUS_ALT) {
                    compatible = false;
                    break;
                }
                if (!p.isFemale() || g != Genotype.HOMOZYGOUS_ALT) continue;
                compatible = false;
                break;
            }
            if (!compatible) continue;
            return true;
        }
        return false;
    }

    public void debugPrint() {
        System.out.println("Pedigree: " + this.familyID + " [n=" + this.getNumberOfIndividualsInPedigree() + "]");
        System.out.println(String.format("Parents: n=%d, Affecteds: n=%d, Unaffecteds: n=%d", this.getNumberOfParentsInPedigree(), this.getNumberOfAffectedsInPedigree(), this.getNumberOfUnaffectedsInPedigree()));
        for (Person p : this.personList) {
            System.out.println(p.getIndex() + ": " + p);
        }
        System.out.println("AffectedList");
        for (Person p : this.affectedList) {
            System.out.println(p.getIndex() + ": " + p);
        }
        System.out.println("ParentList");
        for (Person p : this.parentList) {
            System.out.println(p.getIndex() + ": " + p);
        }
        System.out.println("UnaffectedList");
        for (Person p : this.unaffectedList) {
            System.out.println(p.getIndex() + ": " + p);
        }
    }

    public String getPedigreeSummary() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.familyID).append(":");
        boolean b = false;
        for (Person p : this.personList) {
            if (b) {
                sb.append(":");
            }
            b = true;
            sb.append(p.getIndividualID());
            if (p.isAffected()) {
                sb.append("[affected");
            } else {
                sb.append("[unaffected");
            }
            if (this.parentList.contains(p)) {
                if (p.isFemale()) {
                    sb.append(";mother]");
                    continue;
                }
                sb.append(";father]");
                continue;
            }
            if (p.isFemale()) {
                sb.append(";female]");
                continue;
            }
            sb.append(";male]");
        }
        return sb.toString();
    }

    public boolean singleSampleCompatibleWithAutosomalRecessive(ArrayList<Variant> varList) {
        int n_het = 0;
        for (Variant v : varList) {
            GenotypeCall gc = v.getGenotype();
            Genotype g = gc.getGenotypeInIndividualN(0);
            if (g == Genotype.HOMOZYGOUS_ALT) {
                return true;
            }
            if (g != Genotype.HETEROZYGOUS) continue;
            ++n_het;
        }
        return n_het > 1;
    }

    public boolean singleSampleHasHeterozygousVariant(ArrayList<Variant> varList) {
        for (Variant v : varList) {
            GenotypeCall gc = v.getGenotype();
            Genotype g = gc.getGenotypeInIndividualN(0);
            if (g != Genotype.HETEROZYGOUS) continue;
            return true;
        }
        return false;
    }
}

