package org.genemania.dw.entity;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.TreeMap;
import org.genemania.dw.db.TAIRMirrorTables;
import org.genemania.dw.util.DWUtil;
import org.genemania.dw.util.GenUtil;

/**
 * DW view for a TAIR gene. Encapsulates the attributes of interest
 * for a TAIR gene. For TAIR, the gene ID is actually a locus ID. 
 * Default toString () lists matching EntrezIDs.
 * For display purposes in the toString () method:
 * preference to definition, else to annotation, else to description. Note that
 * Entrez uses the definition as a gene def, but TAIR locus report page uses
 * annotation, instead. The toStringLeftOver () is handy for the leftover TAIR
 * genes in an Entrez based ID mapping file.
 * 
 * @author rashadbadrawi
 */

public class TAIRGene extends ExtResourceGene {

    private String annotation;
    private String description;
    
    public static TreeMap <String, ExtResource> loadAllExt () 
            throws SQLException {

        TreeMap <String, ExtResource> tairMap = TAIRMirrorTables.loadAll();

        return tairMap;
    }
    
    public TAIRGene () {
    
        setSource(ExtResourceGene.RES_TAIR);
        setSpeciesName(DWUtil.SP_CRESS);
    }
    
    @Override
    public void addUniprotID (String uniprotID) {
        
        throw new UnsupportedOperationException ("No implementation for " +
                " Uniprot for TAIR genes.");
        
    }

    @Override
    public void addMGDID (String MGDID) {

        throw new UnsupportedOperationException ("No implementation for " +
                " MGD IDs for TAIR genes.");

    }
    
    public void setAnnotation (String annotation) {
        
        GenUtil.validateString(annotation);
        this.annotation = annotation;
    }
    
    public void setDescription (String description) {
        
        GenUtil.validateString(description);
        this.description = description;
    }
    
    public String getAnnotation () { return annotation; }
    
    public String getDescription () { return description; }
    
    @Override
    public ArrayList <String> getUniprotList () {
        
        throw new UnsupportedOperationException ("No implementation for " +
                " Uniprot for TAIR genes.");
    }

    @Override
    public ArrayList <String> getMGDList () {

        throw new UnsupportedOperationException ("No implementation for " +
                " MGD IDs for TAIR genes.");
    }

    @Override
    public String toString () {
       
        String tempStr = getDBID() + "";
        tempStr += GenUtil.TAB;
        
        tempStr += getID();
        tempStr += GenUtil.TAB;
        
        if (TYPE_PROTEIN_CODING_TAIR.equals (getType ())) {
            tempStr += "True";
        } else {
            tempStr += "False";
        }
        tempStr += GenUtil.TAB;
        
        if (getName() != null) {
            tempStr += getName ();
        } else {
            tempStr += GenUtil.NA;
        }
        tempStr += GenUtil.TAB;
        
        tempStr = GenUtil.addList (tempStr, getRefinedXrefList(LIST_UNIPROT));
        tempStr += GenUtil.TAB;
        
        TreeMap <String, ExtResource> xRefEntMap = 
                getXRef (ExtResource.RES_ENTREZ);
        if (xRefEntMap == null || xRefEntMap.size () == 0) {
            tempStr += GenUtil.NA;
        } else {
            Iterator iterator = xRefEntMap.keySet().iterator();
            boolean firstTime = true;
            while (iterator.hasNext ()) {
                String entGeneID = (String)iterator.next ();
                tempStr += entGeneID;
                if (firstTime) {
                    firstTime = false;
                } else {
                    tempStr += GenUtil.SEMICOLON; 
                }
            }
        }
        tempStr += GenUtil.TAB;
        
        tempStr = GenUtil.addList (tempStr, getRefinedXrefList(LIST_REFSEQ_RNA));
        tempStr += GenUtil.TAB;
        
        tempStr = GenUtil.addList (tempStr, getRefinedXrefList(LIST_REFSEQ_PRO));
        tempStr += GenUtil.TAB;
        
        tempStr = GenUtil.addList (tempStr, getRefinedSynList());
        tempStr += GenUtil.TAB;
        
        if (getDefinition() != null) {
            tempStr += getDefinition();
        } else {
            if (getAnnotation() != null) {
                tempStr += getAnnotation();
            } else {
                if (getDescription() != null) {
                    tempStr += getDescription();
                } else {
                    tempStr += GenUtil.NA;
                }
            }
        }

        return tempStr;
    }

    public String toStringLeftOver (String xRefType) {

        if (!ExtResourceGene.RES_ENTREZ.equals (xRefType)) {
            throw new IllegalArgumentException ("No leftOvers support for " +
                    xRefType);
        }
        String tempStr = getDBID() + "";
        tempStr += GenUtil.TAB;

        tempStr += GenUtil.NA;
        tempStr += GenUtil.TAB;

        if (TYPE_PROTEIN_CODING_TAIR.equals (getType ())) {
            tempStr += "True";
        } else {
            tempStr += "False";
        }
        tempStr += GenUtil.TAB;

        if (getName() != null) {
            tempStr += getName ();
        } else {
            tempStr += GenUtil.NA;
        }
        tempStr += GenUtil.TAB;

        tempStr = GenUtil.addList (tempStr, getRefinedXrefList(LIST_UNIPROT));
        tempStr += GenUtil.TAB;

        tempStr += getID();
        tempStr += GenUtil.TAB;

        tempStr = GenUtil.addList (tempStr, getRefinedXrefList(LIST_REFSEQ_RNA));
        tempStr += GenUtil.TAB;

        tempStr = GenUtil.addList (tempStr, getRefinedXrefList(LIST_REFSEQ_PRO));
        tempStr += GenUtil.TAB;

        tempStr = GenUtil.addList (tempStr, getRefinedSynList());
        tempStr += GenUtil.TAB;

        if (getDefinition() != null) {
            tempStr += getDefinition();
        } else {
            if (getAnnotation() != null) {
                tempStr += getAnnotation();
            } else {
                if (getDescription() != null) {
                    tempStr += getDescription();
                } else {
                    tempStr += GenUtil.NA;
                }
            }
        }

        return tempStr;
    }
}
