package org.baderlab.pdzsvm.data;

import org.biojava.bio.seq.Sequence;

import java.io.File;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.util.*;

import org.baderlab.brain.ProteinProfile;
import org.baderlab.brain.PeptideToProfileReader;
import org.baderlab.brain.ProteinTerminus;

import org.baderlab.pdzsvm.encoding.*;
import org.baderlab.pdzsvm.predictor.pwm.PWM;
import org.baderlab.pdzsvm.utils.PDZSVMUtils;
import org.baderlab.pdzsvm.utils.Constants;
import org.baderlab.pdzsvm.data.manager.DataFileManager;

/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */
public class Data
{
    // Mandatory
    private HashMap domainRawToNumMap;
    private HashMap peptideRawToNumMap;
    private HashMap domainNumToRawMap;
    private HashMap peptideNumToRawMap;

    private List dataList;
    private HashMap domainEncToNumMap;
    private HashMap peptideEncToNumMap;

    private HashMap domainNumToEncMap;
    private HashMap peptideNumToEncMap;

    private int numPos;
    private int numNeg;
    private List posProfileList;
    private List negProfileList;
    private boolean debug = false;

    public Data()
    {
        domainRawToNumMap = new HashMap();
        peptideRawToNumMap = new HashMap();
        domainEncToNumMap = new HashMap();
        peptideEncToNumMap = new HashMap();
        domainNumToRawMap = new HashMap();
        peptideNumToRawMap = new HashMap();
        domainNumToEncMap = new HashMap();
        peptideNumToEncMap = new HashMap();

        dataList = new ArrayList();
        numPos = 0;
        numNeg = 0;
        posProfileList = new ArrayList();
        negProfileList = new ArrayList();
    }

    public void printSummary()
    {
        System.out.println("\t=== SUMMARY (# Interactions) ===");
        System.out.println("\tPositive: " + getNumPositive());
        System.out.println("\tNegative: " + getNumNegative());
        System.out.println("\tTotal: " + getNumInteractions());
        System.out.println();
    }
    public void encodeBindingSiteData(FeatureEncoding domainEncoding, FeatureEncoding peptideEncoding)
    {
        System.out.println("\tEncoding binding site data...");
        Chen16FeatureEncoding bs = new Chen16FeatureEncoding();
        Set keys = domainRawToNumMap.keySet();
        Iterator it = keys.iterator();
        Features encodedDomainFeatures = null;
        int ix = 0;
        while(it.hasNext())
        {
            //System.out.println("\tFeature: " + ix);
            Features features = (Features)it.next();
            String featureString = features.toUndelimitedString();

            Integer domainNum = (Integer)domainRawToNumMap.get(features);

            String organism = "";

            String domainName = "";
            for (int i =0; i < dataList.size();i++)
            {
                Datum dt = (Datum) dataList.get(i);
                if (dt.domainNum == domainNum)
                {
                    domainName = dt.name;
                    organism = dt.organism;
                    break;
                }
            }
            ix = ix+1;
            String bindingSite = bs.getFeatures(featureString,organism);
            encodedDomainFeatures = domainEncoding.encodeFeatures(bindingSite);
            if (encodedDomainFeatures == null)
            {   System.out.println(domainName + ", " + domainNum + "," + features.toUndelimitedString() + "-> NULL");
                continue;
            }

            //System.out.println(id + "->" + encodedFeatures.toString()) ;
            domainNumToEncMap.put(domainNum,encodedDomainFeatures);
            domainEncToNumMap.put(encodedDomainFeatures,domainNum);

        }
        int numDomainEncFeatures = 0;
        if (encodedDomainFeatures != null)
            numDomainEncFeatures = encodedDomainFeatures.numFeatures();
        int numPeptideEncFeatures =0;
        Features encodedPeptideFeatures = null;

        if (peptideEncoding != null)
        {
            keys = peptideRawToNumMap.keySet();
            it = keys.iterator();
            while(it.hasNext())
            {
                Features features = (Features)it.next();
                Integer id = (Integer)peptideRawToNumMap.get(features);
                encodedPeptideFeatures = peptideEncoding.encodeFeatures(features.toUndelimitedString());
                peptideNumToEncMap.put(id,encodedPeptideFeatures);
                peptideEncToNumMap.put(encodedPeptideFeatures,id);
            }

            numPeptideEncFeatures = encodedPeptideFeatures.numFeatures();


        }
        System.out.println("\tDomain encoding used: " + domainEncoding.getEncodingName()+ ", Num Features: " + encodedDomainFeatures.numFeatures());

        System.out.println("\tPeptide encoding used: " + peptideEncoding.getEncodingName()+ ", Num Features: " + encodedPeptideFeatures.numFeatures());

        System.out.println("\tDomain Map/Encoded Domain List Size:" + domainRawToNumMap.size() + "-" + domainNumToEncMap.size());
        System.out.println("\tPeptide Raw Map/Peptide Encoded List Size:" + peptideRawToNumMap.size() + "-" + peptideEncToNumMap.size());
        System.out.println("\tNumber of encoded features: " + numDomainEncFeatures + ", " + numPeptideEncFeatures);
        System.out.println();
    }
    public void encodeData(FeatureEncoding domainEncoding, FeatureEncoding peptideEncoding)
    {
        System.out.println("\n\tEncoding data...");

        Set keys = domainRawToNumMap.keySet();
        Iterator it = keys.iterator();
        Features encodedFeatures = null;
        while(it.hasNext())
        {
            Features features = (Features)it.next();

            Integer id = (Integer)domainRawToNumMap.get(features);

            //System.out.println(id + ":" + features.toUndelimitedString());

            encodedFeatures = domainEncoding.encodeFeatures(features.toUndelimitedString());

            if (encodedFeatures == null)
            {
                String domainName = "";
                for (int i =0; i < dataList.size();i++)
                {
                    Datum dt = (Datum) dataList.get(i);
                    if (dt.domainNum == id)
                    {
                        domainName = dt.name;
                        break;
                    }
                }
                System.out.println(domainName + ", " + id + "," + features.toUndelimitedString() + "-> NULL");
                continue;
            }

            //System.out.println(id + "->" + encodedFeatures.toString()) ;
            domainNumToEncMap.put(id,encodedFeatures);
            domainEncToNumMap.put(encodedFeatures,id);

        }
        int numEncFeatures = 0;
        if (encodedFeatures != null)
            numEncFeatures = encodedFeatures.numFeatures();

        if (debug)
            System.out.println("\tDomain encoding used: " + domainEncoding.getEncodingName() + ", Num Features: " + numEncFeatures);

        if (peptideEncoding != null)
        {
            keys = peptideRawToNumMap.keySet();
            it = keys.iterator();
            while(it.hasNext())
            {
                Features features = (Features)it.next();
                Integer id = (Integer)peptideRawToNumMap.get(features);
                encodedFeatures = peptideEncoding.encodeFeatures(features.toUndelimitedString());
                peptideNumToEncMap.put(id,encodedFeatures);
                peptideEncToNumMap.put(encodedFeatures,id);
            }
            System.out.println("\tPeptide encoding used: " + peptideEncoding.getEncodingName()+ ", Num Features: " + encodedFeatures.numFeatures());

                System.out.println("\tDomain Map/Encoded Domain List Size:" + domainRawToNumMap.size() + "-" + domainEncToNumMap.size());
                System.out.println("\tPeptide Raw Map/Peptide Encoded List Size:" + peptideRawToNumMap.size() + "-" + peptideEncToNumMap.size());
            System.out.println("\t Number of encoded features: " + numEncFeatures);
        }
    }

    public int getNumPositive()
    {
        return numPos;
    }
    public int getNumNegative()
    {
        return numNeg;
    }
    public int getNumInteractions()
    {
        return numPos + numNeg;
    }
    public boolean isEmpty()
    {
        return dataList.isEmpty();
    }
    public int getNumDomains()
    {
        return domainRawToNumMap.size();
    }
    public int getNumPeptides()
    {
        return peptideRawToNumMap.size();
    }
    public HashMap getDomainRawToNumMap()
    {
        return domainRawToNumMap;
    }

    public HashMap getPeptideRawToNumMap()
    {
        return peptideRawToNumMap;
    }
    public HashMap getDomainNumToRawMap()
    {
        return domainNumToRawMap;
    }
    public HashMap getPeptideNumToRawMap()
    {
        return peptideNumToRawMap;
    }
    public List getDataList()
    {
        return dataList;
    }
    public List[] getClassSplitDataList()
    {
        List posDataList = new ArrayList();
        List negDataList = new ArrayList();

        for (int i=0; i < dataList.size();i++)
        {
            Datum dt = (Datum)dataList.get(i);
            String posorneg = dt.posneg;
            if (posorneg.equals(Constants.CLASS_YES))
            {
                posDataList.add(dt);
            }
            else
            {
                negDataList.add(dt);
            }

        }
        List[] posNegDataListArray = new ArrayList[2];
        posNegDataListArray[0] = posDataList;
        posNegDataListArray[1] =negDataList;
        return posNegDataListArray;
    }
    public HashMap getDomainEncToNumMap()
    {
        return domainEncToNumMap;
    }
    public HashMap getPeptideEncToNumMap()
    {
        return peptideEncToNumMap;
    }

    public HashMap getDomainNumToEncMap()
    {
        return domainNumToEncMap;
    }
    public HashMap getPeptideNumToEncMap()
    {
        return peptideNumToEncMap;
    }

    // Makes new data
    public void addRawData(List dataList, HashMap domainNumToRawMap, HashMap peptideNumToRawMap)
    {
        this.dataList = dataList;
        for (int i=0; i < dataList.size();i++)
        {
            Datum dt = (Datum) dataList.get(i);
            int domainNum = dt.domainNum;
            int peptideNum = dt.peptideNum;

            Features domainAAFeatures = (Features)domainNumToRawMap.get(domainNum);
            Features peptideAAFeatures = (Features)peptideNumToRawMap.get(peptideNum);

            this.domainRawToNumMap.put(domainAAFeatures, domainNum);
            this.peptideRawToNumMap.put(peptideAAFeatures, peptideNum);
            this.domainNumToRawMap.put(domainNum, domainAAFeatures);
            this.peptideNumToRawMap.put(peptideNum, peptideAAFeatures);

            if (dt.posneg.equals(Constants.CLASS_YES))
                numPos = numPos + 1;
            else
                numNeg = numNeg + 1;
        }
    }
    // Makes new data     
    public void addRawData(List domainList, List peptideList, List domainNamesList, String expMethod, String organism, String posneg)
    {
        AminoAcidFeatureEncoding rawAAEncoding = new AminoAcidFeatureEncoding();

        for (int i=0;i < peptideList.size();i++)
        {
            String peptide = (String)peptideList.get(i);
            Features peptideAAFeatures = rawAAEncoding.encodeFeatures(peptide);
            peptideNumToRawMap.put(i,peptideAAFeatures);
            peptideRawToNumMap.put(peptideAAFeatures, i);

        }
        for (int i=0;i < domainList.size();i++)
        {
            String domain = (String)domainList.get(i);
            Features domainAAFeatures = rawAAEncoding.encodeFeatures(domain);
            domainNumToRawMap.put(i,domainAAFeatures);
            domainRawToNumMap.put(domainAAFeatures, i);

        }

        for (int i=0; i < domainList.size();i++)
        {
            String name = (String)domainNamesList.get(i);

            for (int j=0; j < peptideList.size();j++)
            {
                Datum dh = new Datum(name, expMethod, organism, posneg, i, j);
                dataList.add(dh);
            }
        }


    }


    public void addRawData(List proteinProfileList, String posneg)
    {
        //System.out.println("Loading data for: " + projectFileName);
        if (posneg.equals(Constants.CLASS_YES))
        {
            posProfileList.addAll(proteinProfileList);
        }
        else
        {
            negProfileList.addAll(proteinProfileList);

        }

        AminoAcidFeatureEncoding rawAAEncoding = new AminoAcidFeatureEncoding();
        int domainsAdded = 0;
        int peptidesAdded = 0;
        int numPosAdded = 0;
        int numNegAdded = 0;
        boolean printHeading = false;
        // Iterate over each protein profile and add to data list
        for (int i = 0; i < proteinProfileList.size(); i++)
        {
            ProteinProfile proteinProfile = (ProteinProfile) proteinProfileList.get(i);

            String name = proteinProfile.getName();
            String organism = proteinProfile.getOrganism();
            String expMethod = PDZSVMUtils.methodLongToShortForm(proteinProfile.getExperimentalMethod());

            if (printHeading)
            {
                System.out.println("\n\t=== " +
                        PDZSVMUtils.organismLongToShortForm(organism) + " " +
                        expMethod + " " + posneg + " ===");
                printHeading = false;
            }
            String domainSequence = proteinProfile.getDomainSequence();
            Collection peptideCollection = proteinProfile.getSequenceMap();
            Iterator it = peptideCollection.iterator();

            // if this profile has no peptide sequences don't add it
            if (!it.hasNext())
            {
                System.out.println("\tProfile " +  proteinProfile.getName() + " has no peptides.  Not added.");
                continue;
            }
            // Encode the domain squence
            Features domainAAFeatures = rawAAEncoding.encodeFeatures(domainSequence);
            Integer domainNumInteger = (Integer)domainRawToNumMap.get(domainAAFeatures);
            if (domainNumInteger == null)
            {
                // Put the new domain into the domain raw map
                domainRawToNumMap.put(domainAAFeatures,domainRawToNumMap.size());
                domainNumInteger = (Integer)domainRawToNumMap.get(domainAAFeatures);
                domainsAdded = domainsAdded+1;
            }
            int domainNum = domainNumInteger.intValue();

            while(it.hasNext())
            {
                Sequence peptideSeq = (Sequence)it.next();
                String peptideSequence = peptideSeq.seqString();

                // Encode the peptide using amino acid encoding
                Features peptideAAFeatures = rawAAEncoding.encodeFeatures(peptideSequence);
                Integer peptideNumInteger = (Integer)peptideRawToNumMap.get(peptideAAFeatures);
                if (peptideNumInteger == null)
                {
                    // Put the peptide into the peptide raw map
                    peptideRawToNumMap.put(peptideAAFeatures,peptideRawToNumMap.size());
                    peptideNumInteger = (Integer)peptideRawToNumMap.get(peptideAAFeatures);
                    peptidesAdded = peptidesAdded+1;
                }
                int peptideNum = peptideNumInteger.intValue();

                //System.out.println( name + "\t" + method + "\t" + organism + "\t" + posneg + "\t" + domainNum + "\t" + peptideNum);

                // Put the new domain-peptide interaction into a Datum object
                Datum dh = new Datum(name, expMethod, organism, posneg, domainNum, peptideNum);
                //System.out.println("*" + dh.toString());
                if (dh.posneg.equals(Constants.CLASS_YES))
                    numPosAdded = numPosAdded +1;
                else
                    numNegAdded = numNegAdded + 1;

                // Add the new datum to the data list
                dataList.add(dh);
            } // while
            // } // if
        } //for each profile
        numPos = numPos + numPosAdded;
        numNeg = numNeg + numNegAdded;
        //System.out.println("\tNum domains added: " + domainsAdded + ", total " + domainRawMap.size());
        //System.out.println("\tNum peptides added: " + peptidesAdded + ", total " + peptideRawMap.size());
        //System.out.println("\tNum positive interactions added: " + numPosAdded + ", total " + numPos);
        //System.out.println("\tNum negative interactions added:  " + numNegAdded + ", total " + numNeg);


        domainNumToRawMap = reverseMap(domainRawToNumMap);
        peptideNumToRawMap = reverseMap(peptideRawToNumMap);

    }
    private static HashMap reverseMap(HashMap keyToDataMap)
    {
        HashMap dataToKeyMap = new HashMap();
        Set keys = keyToDataMap.keySet();
        Iterator it = keys.iterator();
        while(it.hasNext())
        {
            Features keyFeatures = (Features) it.next();
            Integer dataNum = (Integer)keyToDataMap.get(keyFeatures);
            dataToKeyMap.put(dataNum, keyFeatures);
        }
        return dataToKeyMap;
    }

    public List getExpMethods()
    {
        List methodList = new ArrayList();
        for (int i=0; i < dataList.size();i++)
        {
            Datum dt = (Datum) dataList.get(i);
            if (!methodList.contains(dt.expMethod))
                methodList.add(dt.expMethod);
        }
        return methodList;
    }
    public List getDomainNames()
    {
        List domainNamesList = new ArrayList();
        for (int i=0; i < dataList.size();i++)
        {
            Datum dt = (Datum) dataList.get(i);
            if (!domainNamesList.contains(dt.name))
                domainNamesList.add(dt.name);
        }
        return domainNamesList;
    }
    public List getPeptideNums()
    {
        List peptideNumsList = new ArrayList();
        for (int i=0; i < dataList.size();i++)
        {
            Datum dt = (Datum) dataList.get(i);
            if (!peptideNumsList.contains(dt.peptideNum))
                peptideNumsList.add(dt.peptideNum);
        }
        return peptideNumsList;
    }
    public Features[] getRawDataPair(Datum dt)
    {
        int domainNum = dt.domainNum;
        int peptideNum = dt.peptideNum;

        HashMap domainNumToRawMap = getDomainNumToRawMap();
        HashMap peptideNumToRawMap = getPeptideNumToRawMap();


        Features[] featurePairArray = new Features[]{null, null};

        if (domainNumToRawMap.size() >0)
        {
            Features domainAAFeatures = (Features)domainNumToRawMap.get(domainNum);
            featurePairArray[0] = domainAAFeatures;
        }
        if (peptideNumToRawMap.size() > 0)
        {
            Features peptideAAFeatures = (Features)peptideNumToRawMap.get(peptideNum);
            featurePairArray[1] = peptideAAFeatures;

        }
        return featurePairArray;
    }

    public  Features getEncodedDataPair(Datum dt)
    {
        int domainNum = dt.domainNum;
        int peptideNum = dt.peptideNum;
        Features encodedDomainPeptideFeatures = new Features();

        // look up encoded features for this pair
        if (domainNumToEncMap.size() >0)
        {
            encodedDomainPeptideFeatures.addFeatures((Features)domainNumToEncMap.get(domainNum));
        }

        if (peptideNumToEncMap.size() >0)
        {
            //System.out.println("peptide encoded data size: " + peptideNum + ", " + peptideEncodedList.size());
            encodedDomainPeptideFeatures.addFeatures((Features)peptideNumToEncMap.get(peptideNum));
        }
        return encodedDomainPeptideFeatures;
    }

    public static void main(String[] args)
    {
        String projectFileName = "/Users/shirleyhui/Data/SVMProject/Data/Worm/PDZ/ProjectFiles/projectFileWorm.txt";
        File projectFile = new File(projectFileName);
        File codonBiasFile = new File(DataFileManager.NNK_CODON_BIAS_FILENAME);
        List profileList = PeptideToProfileReader.readPeptidesAsProfiles(projectFile,7, ProteinTerminus.C,0.01,codonBiasFile,true,true);
        for (int i = 0 ; i < profileList.size();i++)
        {
            ProteinProfile proteinProfile = (ProteinProfile) profileList.get(i);
            String name = proteinProfile.getName();

            PWM pwm = new PWM(proteinProfile);

            Collection peptideCollection = proteinProfile.getSequenceMap();
            Iterator it = peptideCollection.iterator();
            int ii=0;

        }

    }
   
    public void saveToLibSVMFormat(String libSVMFileName)
    {
        try
        {
            BufferedWriter bw = new BufferedWriter(new FileWriter(new File(libSVMFileName)));
            for (int i = 0; i < dataList.size();i++)
            {
                String outString = "";

                //System.out.print(".");
                Datum dt= (Datum)dataList.get(i);
                if (dt.posneg.equals(Constants.CLASS_YES))
                    outString = outString + "+1";
                else
                    outString = outString + "-1";
                Features ft = getEncodedDataPair(dt);

                //Features ft = getEncodedContactDataPair(dt);
                String[] splitft = ft.toString().split(",");
                for (int j = 0; j < splitft.length;j++)
                {
                    outString  = outString + " " + (j+1) +":" + splitft[j];
                }
                //outString = outString + "\n";
                bw.write(outString);
                bw.write("\n");
                //System.out.println(outString);

            }
            //bw.write(outString);
            bw.close();
            System.out.println();
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }
    }

    public void saveToArffFormat(String arffFileName)
    {
        try
        {
            String arff = "% ARFF file for binding data\n" +
                    "%\n" +
                    "@relation bind\n";
            BufferedWriter bw = new BufferedWriter(new FileWriter(new File(arffFileName)));
            for (int i = 0; i < dataList.size();i++)
            {

                Datum dt= (Datum)dataList.get(i);
                Features ft = getEncodedDataPair(dt);
                if (i==0)
                {
                    for (int j = 0;j < ft.numFeatures();j++)
                    {
                        arff = arff + "@attribute feature_"+j + " numeric\n";
                    }
                    arff = arff + "@attribute bind? {yes, no}\n" +
                            "@data\n" +
                            "%\n";
                }
                arff = arff + ft.toString();
                if (dt.posneg.equals(Constants.CLASS_YES))
                    arff = arff+",yes\n";
                else
                    arff = arff+",no\n";
            }
            System.out.println(arff);
            bw.write(arff);
            bw.close();
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }

    }

    public void saveToFeatureFormat(String fileName)
    {
        try
        {
            String arff = "";
            BufferedWriter bw = new BufferedWriter(new FileWriter(new File(fileName)));
            for (int i = 0; i < dataList.size();i++)
            {

                Datum dt= (Datum)dataList.get(i);
                Features ft = getEncodedDataPair(dt);

                // parse out commas
                String featString = ft.toString();
                String[] splitString = featString.split(",");
                String featStringSpaced = "";
                for (int ii =0; ii < splitString.length;ii++)
                {
                    if(ii==0)
                        featStringSpaced = splitString[ii];
                    else
                        featStringSpaced = featStringSpaced + " "+splitString[ii];
                }
                /*if (dt.posneg.equals(Constants.CLASS_YES))
                    arff = "1";
                else
                    arff = "-1";
                    */
                //arff = featStringSpaced + "\n";
                //System.out.println(arff);
                bw.write(featStringSpaced +"\n");

            }
            bw.close();
        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
        }

    }
    public void print()
    {
        System.out.println("\tPrint Data...");
        System.out.println("\tTotal number instances: " + dataList.size());
        // print data Map out
        for(int i =0;i < dataList.size();i++)
        {
            Datum dh = (Datum)dataList.get(i);
            System.out.println(dh);
        }
        System.out.println("\tTotal number domains: " + domainRawToNumMap.size());
        Set domainNumSet = domainNumToRawMap.keySet();
        List domainNumList = new ArrayList(domainNumSet);
        Collections.sort(domainNumList);

        for (int i =0;i < domainNumList.size();i++)
        {
            int domainNum = (Integer)domainNumList.get(i);
            Features domainFeatures = (Features)domainNumToRawMap.get(domainNum);
            System.out.println(domainNum + "\t" + domainFeatures.toString());
        }

        System.out.println("\tTotal number peptides: " + peptideRawToNumMap.size());
        Set peptideNumSet = peptideNumToRawMap.keySet();
        List peptideNumList = new ArrayList(peptideNumSet);
        Collections.sort(peptideNumList);
        for (int i =0;i < peptideNumList.size();i++)
        {
            int peptideNum = (Integer)peptideNumList.get(i);
            Features peptideFeatures = (Features)peptideNumToRawMap.get(peptideNum);
            System.out.println(peptideNum + "\t" + peptideFeatures.toString());
        }

        System.out.println("\tTotal number Encoded domains: " + domainEncToNumMap.size());

        for (int i =0;i < domainNumToEncMap.size();i++)
        {
            Features domainFeatures = (Features)domainNumToEncMap.get(i);
            //System.out.println(i + "\t" + domainFeatures.toString());
        }

        System.out.println("\tTotal number Encoded peptides: " + peptideEncToNumMap.size());

        for (int i =0;i < peptideNumToEncMap.size();i++)
        {
            Features peptideFeatures = (Features)peptideNumToEncMap.get(i);
            //System.out.println(i + "\t" + peptideFeatures.toString());

        }
    }
}