package   org.baderlab.pdzsvm.analysis;

import cytoscape.data.annotation.Annotation;
import cytoscape.data.annotation.Ontology;
import cytoscape.task.util.TaskManager;
import cytoscape.task.ui.JTaskConfig;

import java.util.*;
import java.math.BigDecimal;
import java.io.*;
import java.text.DecimalFormat;

import BiNGO.*;
import org.baderlab.pdzsvm.utils.Constants;

/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * This file calls Bingo without GUI
 */
public class Bingo {
    private BingoParameters params;
    private String organism ="";
    private HashMap<String,HashSet<String>> redundantIDs = new HashMap<String,HashSet<String>>() ;
    static private double CUTOFF = 0.05;
    private String bingoDir = "jar:file:/Applications/Cytoscape_v2.6.0/plugins/BiNGO.jar!/";

    public Bingo(String organism, String outputDir)
    {
        String ontologyFileName = bingoDir+"GO_Biological_Process";

        String species = "";
        String annotOrganismFileName;
        if (organism.equals(Constants.HUMAN))
        {
            this.organism = "Human";
            species = "Homo Sapiens";
            annotOrganismFileName= bingoDir+"H_sapiens_default";
        }
        else if (organism.equals(Constants.FLY))
        {
            this.organism = "Fly";

            species = "Drosophila_melanogaster";
            annotOrganismFileName= bingoDir+"D_melanogaster_default";
        }
        else if (organism.equals(Constants.WORM))
        {
            this.organism = "Worm";

            species = "Caenorhabditis_elegans";
            annotOrganismFileName= bingoDir+"C_elegans_default";
        }
        else
        {
            return;
        }

        try
        {
            params = new BingoParameters(bingoDir +"/");
        }
        catch(Exception e)
        {
            System.out.println("Exception: "+ e);
            e.printStackTrace();
        }
        params.setVisualization("No Visualization");
        params.setSpecies(species);
        params.setTextOrGraph(true);
        params.setFileoutput(true);
        params.setFileoutput_dir(outputDir);
        params.setOverOrUnder("Overrepresentation");
        params.setSignificance(new BigDecimal(new String("0.05")));  // not really used since category is CATEGORY

        params.setReferenceSet(BingoAlgorithm.GENOME);
        params.setCategory(BingoAlgorithm.CATEGORY);
        params.setAnnotation_default(true);
        params.setOntology_default(true);
        params.setOntologyFile(ontologyFileName);
        params.setAnnotationFile(annotOrganismFileName);
        HashSet deleteEvidenceCodes = new HashSet();
        // Don't use annotations that are made automatically or by sequence similarity
        deleteEvidenceCodes.add("IEA");
        deleteEvidenceCodes.add("ISS");

        params.setDeleteCodes(deleteEvidenceCodes);
        AnnotationParser annParser = params.initializeAnnotationParser();
        JTaskConfig config = new JTaskConfig();
        TaskManager.executeTask(annParser, config);

        Annotation annotation = annParser.getAnnotation();
        Ontology ontology = annParser.getOntology();
        HashMap alias = annParser.getAlias();

        params.setOntology(ontology);
        params.setAlias(alias);
        params.setAnnotation(annotation);
    }

    public void analyze(String ensemblIds, String domainName )
    {
        HashSet noClassificationSet = new HashSet();

        HashMap testMap = null;
        HashMap correctionMap = null;
        HashMap mapSmallX = null;
        HashMap mapSmallN = null;
        HashMap mapBigX = null;
        HashMap mapBigN = null;
        params.setCluster_name(domainName);
        params.setTextInput(ensemblIds);
        HashSet selectedNodes = getSelectedCanonicalNamesFromTextArea();

        params.setSelectedNodes(selectedNodes);
        params.setAllNodes(getAllCanonicalNamesFromAnnotation(selectedNodes));

        int [] testData = getClassificationsFromVector(selectedNodes,noClassificationSet);

        BingoAlgorithm algorithm = new BingoAlgorithm(params);

        CalculateTestTask test = algorithm.calculate_distribution();

        JTaskConfig config = new JTaskConfig();
        TaskManager.executeTask(test, config);
        testMap = test.getTestMap();

        CalculateCorrectionTask correction = algorithm.calculate_corrections(testMap);
        if ((correction != null)&&(!params.getTest().equals(BingoAlgorithm.NONE)))
        {
            TaskManager.executeTask(correction, config);
            correctionMap = correction.getCorrectionMap();
        }

        mapSmallX = test.getMapSmallX();
        mapSmallN = test.getMapSmallN();
        mapBigX = test.getMapBigX();
        mapBigN = test.getMapBigN();

        if (params.isFileoutput())
        {
            CreateBiNGOFile    file = new CreateBiNGOFile(testMap,
                    correctionMap,
                    mapSmallX,
                    mapSmallN,
                    mapBigX,
                    mapBigN,
                    params.getSignificance().toString(),
                    params.getAnnotation(),
                    params.getDeleteCodes(),
                    params.getAlias(),
                    params.getOntology(),
                    params.getAnnotationFile().toString(),
                    params.getOntologyFile().toString(),
                    params.getTest() + "",
                    params.getCorrection() + "",
                    params.getOverOrUnder() + "",
                    params.getFileoutput_dir(),
                    params.getCluster_name() + "-"+organism+".bgo",
                    params.getReferenceSet() + "",
                    params.getCategory() + "",
                    selectedNodes,
                    noClassificationSet);

            file.makeFile();
        }
    }

    /**
     * Method that gets the classifications from a HashSet of canonical names.
     *
     * @param canonicalNameVector HashSet of canonical names.
     * @return int[] classifications.
     */
    public int [] getClassificationsFromVector(HashSet canonicalNameVector, HashSet noClassificationsSet) {
        HashSet classificationsVector = new HashSet();
        HashMap<String,HashSet<String>> alias = params.getAlias();
        // array for go labels.
        int [] goLabelsName;
        Iterator it2 = canonicalNameVector.iterator();
        while(it2.hasNext()){
            String name = it2.next() + "";
            HashSet identifiers = alias.get(name) ;
            HashSet cls = new HashSet();
            // array for go labels.
            if(identifiers != null){
                Iterator it = identifiers.iterator() ;
                while(it.hasNext()){
                    goLabelsName = params.getAnnotation().getClassifications(it.next() + "");
                    for (int t = 0; t < goLabelsName.length; t++){
                        cls.add(goLabelsName[t] + "");
                    }
                }
            }
            if(cls.size() == 0){
                noClassificationsSet.add(name);
            }
            Iterator it3 = cls.iterator();
            while(it3.hasNext()){
                classificationsVector.add(it3.next() + "");
            }
        }
        int [] classifications = new int [classificationsVector.size()];
        it2 = classificationsVector.iterator();
        int i = 0;
        while(it2.hasNext()){
            classifications[i] = Integer.parseInt(it2.next() + "");
            i++;
        }
        return classifications;
    }
    /**
     * method that gets the canonical names for the whole annotation.
     *
     * @return HashSet containing the canonical names.
     */
    public HashSet getAllCanonicalNamesFromAnnotation(HashSet selectedNodes) {
        String [] nodes = params.getAnnotation().getNames();
        // HashSet for storing the canonical names
        HashSet canonicalNameVector = new HashSet();
        for (int i = 0; i < nodes.length; i++) {
            if (nodes[i] != null && (nodes[i].length() != 0)){
                canonicalNameVector.add(nodes[i].toUpperCase());
            }
        }

        //replace canonical names in reference set that match one of the canonical names in the selected cluster, to get rid of e.g. splice variants if the non-splice-specific gene is part of the selection, and to avoid conflicts between names in ref set and selection
        HashMap<String,HashSet<String>> alias = params.getAlias();
        Iterator it2 = selectedNodes.iterator();
        while(it2.hasNext()){
            String name = it2.next() + "" ;
            HashSet tmp = alias.get(name);
            if(tmp != null){
                Iterator it = tmp.iterator();
                while(it.hasNext()){
                    canonicalNameVector.remove(it.next() + "");
                }
                //add selected node name
                canonicalNameVector.add(name);
            }
        }
        return canonicalNameVector;
    }
    /**
     * method that gets the canonical names from text input.
     *
     * @return HashSet containing the canonical names.
     */
    public HashSet getSelectedCanonicalNamesFromTextArea() {
        String textNodes = params.getTextInput();
        String [] nodes = textNodes.split("\\s+");
        // HashSet for storing the canonical names
        HashSet canonicalNameVector = new HashSet();
        HashSet<HashSet<String>> mapNames = new HashSet<HashSet<String>>();
        // iterate over every node view to get the canonical names.
        for (int i = 0; i < nodes.length; i++) {
            if (nodes[i] != null && nodes[i].length() != 0 && !canonicalNameVector.contains(nodes[i].toUpperCase())){
                if(mapNames.contains(params.getAlias().get(nodes[i].toUpperCase()))){
                    redundantIDs.put(nodes[i].toUpperCase(), (HashSet<String>) params.getAlias().get(nodes[i].toUpperCase()));

                }
                if(params.getAlias().get(nodes[i].toUpperCase()) != null){
                    mapNames.add((HashSet<String>) params.getAlias().get(nodes[i].toUpperCase()));
                }
                canonicalNameVector.add(nodes[i].toUpperCase());
            }
        }
        return canonicalNameVector;
    }

    public static void main(String[] args)
    {
        String bingoOutDir = "/Users/shirleyhui/Desktop/Results/Bingo-1/";

        /*//String ensemblIds = "ENSEMBL:ENSG00000165637 ENSEMBL:ENSG00000183128";
        //String ensemblIds = "Ensembl:ENSG00000131591 Ensembl:ENSG00000177807 Ensembl:ENSG00000147246 Ensembl:ENSG00000070087 Ensembl:ENSG00000163870 Ensembl:ENSG00000006715 Ensembl:ENSG00000169562 Ensembl:ENSG00000235046 Ensembl:ENSG00000174898 Ensembl:ENSG00000171303 Ensembl:ENSG00000130702 Ensembl:ENSG00000186654 Ensembl:ENSG00000182489 Ensembl:ENSG00000169499 Ensembl:ENSG00000197887 Ensembl:ENSG00000172774 Ensembl:ENSG00000102057 Ensembl:ENSG00000214226 Ensembl:ENSG00000184408 Ensembl:ENSG00000153339 Ensembl:ENSG00000124523 Ensembl:ENSG00000133020 Ensembl:ENSG00000169122 Ensembl:ENSG00000111077 Ensembl:ENSG00000165215 Ensembl:ENSG00000154229 Ensembl:ENSG00000005448 Ensembl:ENSG00000131951 Ensembl:ENSG00000001460 Ensembl:ENSG00000135973 Ensembl:ENSG00000155561 Ensembl:ENSG00000166508 Ensembl:ENSG00000085276 Ensembl:ENSG00000244102 Ensembl:ENSG00000177565 Ensembl:ENSG00000243489 Ensembl:ENSG00000127824 Ensembl:ENSG00000183473 Ensembl:ENSG00000241553 Ensembl:ENSG00000227782 Ensembl:ENSG00000153498 Ensembl:ENSG00000180834 Ensembl:ENSG00000119408 Ensembl:ENSG00000197535 Ensembl:ENSG00000172175 Ensembl:ENSG00000173933 Ensembl:ENSG00000173914 Ensembl:ENSG00000155438 Ensembl:ENSG00000118257 Ensembl:ENSG00000102468 Ensembl:ENSG00000168412 Ensembl:ENSG00000187607 Ensembl:ENSG00000128534 Ensembl:ENSG00000185033 Ensembl:ENSG00000078795 Ensembl:ENSG00000171385 Ensembl:ENSG00000241839 Ensembl:ENSG00000141338 Ensembl:ENSG00000124615 Ensembl:ENSG00000197461 Ensembl:ENSG00000148219 Ensembl:ENSG00000144481 Ensembl:ENSG00000183155 Ensembl:ENSG00000136895 Ensembl:ENSG00000197533 Ensembl:ENSG00000144061 Ensembl:ENSG00000153234 Ensembl:ENSG00000151773 Ensembl:ENSG00000156500 Ensembl:ENSG00000004766 Ensembl:ENSG00000163605 Ensembl:ENSG00000197584 Ensembl:ENSG00000213952 Ensembl:ENSG00000131650 Ensembl:ENSG00000117614 Ensembl:ENSG00000214188 Ensembl:ENSG00000179583 Ensembl:ENSG00000105287 Ensembl:ENSG00000206531 Ensembl:ENSG00000100296 Ensembl:ENSG00000241878 Ensembl:ENSG00000003147 Ensembl:ENSG00000198014 Ensembl:ENSG00000212725 Ensembl:ENSG00000214518 Ensembl:ENSG00000212724 Ensembl:ENSG00000213417 Ensembl:ENSG00000237368 Ensembl:ENSG00000075391 Ensembl:ENSG00000155542 Ensembl:ENSG00000182858 Ensembl:ENSG00000172243 Ensembl:ENSG00000188674 Ensembl:ENSG00000152463 Ensembl:ENSG00000196976 Ensembl:ENSG00000163362 Ensembl:ENSG00000084731 Ensembl:ENSG00000160539 Ensembl:ENSG00000115107 Ensembl:ENSG00000124253 Ensembl:ENSG00000158987 Ensembl:ENSG00000109756 Ensembl:ENSG00000103043 Ensembl:ENSG00000157514 Ensembl:ENSG00000009724 Ensembl:ENSG00000126368 Ensembl:ENSG00000196123 Ensembl:ENSG00000175768 Ensembl:ENSG00000142687 Ensembl:ENSG00000118193 Ensembl:ENSG00000048462 Ensembl:ENSG00000178996 Ensembl:ENSG00000114491 Ensembl:ENSG00000206530 Ensembl:ENSG00000163702 Ensembl:ENSG00000087085 Ensembl:ENSG00000141027 Ensembl:ENSG00000140481 Ensembl:ENSG00000178425 Ensembl:ENSG00000106541 Ensembl:ENSG00000146373 Ensembl:ENSG00000131188 Ensembl:ENSG00000196335 Ensembl:ENSG00000107679 Ensembl:ENSG00000106100 Ensembl:ENSG00000071282 Ensembl:ENSG00000124209 Ensembl:ENSG00000160949 Ensembl:ENSG00000174015 Ensembl:ENSG00000177692 Ensembl:ENSG00000230413 Ensembl:ENSG00000206506 Ensembl:ENSG00000235346 Ensembl:ENSG00000204632 Ensembl:ENSG00000233095 Ensembl:ENSG00000235680 Ensembl:ENSG00000237216 Ensembl:ENSG00000119698 Ensembl:ENSG00000114993 Ensembl:ENSG00000100364 Ensembl:ENSG00000112297 Ensembl:ENSG00000011114 Ensembl:ENSG00000145075 Ensembl:ENSG00000213937 Ensembl:ENSG00000134780 Ensembl:ENSG00000135502 Ensembl:ENSG00000203359 Ensembl:ENSG00000185049 Ensembl:ENSG00000186132 Ensembl:ENSG00000160219 Ensembl:ENSG00000183765 Ensembl:ENSG00000179409 Ensembl:ENSG00000183695 Ensembl:ENSG00000147133 Ensembl:ENSG00000106070 Ensembl:ENSG00000155744 Ensembl:ENSG00000144048 Ensembl:ENSG00000185104 Ensembl:ENSG00000160633 Ensembl:ENSG00000069424 Ensembl:ENSG00000151577 Ensembl:ENSG00000197183 Ensembl:ENSG00000161381 Ensembl:ENSG00000133116 Ensembl:ENSG00000177352 Ensembl:ENSG00000164761 Ensembl:ENSG00000123243 Ensembl:ENSG00000155761 Ensembl:ENSG00000178919 Ensembl:ENSG00000171298 Ensembl:ENSG00000174606 Ensembl:ENSG00000148377 Ensembl:ENSG00000089163 Ensembl:ENSG00000175485 Ensembl:ENSG00000118960 Ensembl:ENSG00000135315 Ensembl:ENSG00000184304 Ensembl:ENSG00000151136 Ensembl:ENSG00000169202 Ensembl:ENSG00000204070 Ensembl:ENSG00000119487 Ensembl:ENSG00000134815 Ensembl:ENSG00000140563 Ensembl:ENSG00000235396 Ensembl:ENSG00000206312 Ensembl:ENSG00000238196 Ensembl:ENSG00000204301 Ensembl:ENSG00000234876 Ensembl:ENSG00000232339 Ensembl:ENSG00000223355 Ensembl:ENSG00000150403 Ensembl:ENSG00000149930 Ensembl:ENSG00000084754 Ensembl:ENSG00000166862 Ensembl:ENSG00000075461 Ensembl:ENSG00000006116 Ensembl:ENSG00000168256 Ensembl:ENSG00000166938 Ensembl:ENSG00000024422 Ensembl:ENSG00000183486 Ensembl:ENSG00000071127 Ensembl:ENSG00000111404 Ensembl:ENSG00000169427 Ensembl:ENSG00000109586 Ensembl:ENSG00000165406 Ensembl:ENSG00000165097 Ensembl:ENSG00000129824 Ensembl:ENSG00000157828 Ensembl:ENSG00000198034 Ensembl:ENSG00000205930 Ensembl:ENSG00000139132 Ensembl:ENSG00000145040 Ensembl:ENSG00000225190 Ensembl:ENSG00000233758 Ensembl:ENSG00000114757 Ensembl:ENSG00000130413 Ensembl:ENSG00000117500 Ensembl:ENSG00000174547 Ensembl:ENSG00000105227 Ensembl:ENSG00000132950 Ensembl:ENSG00000166436 Ensembl:ENSG00000198312 Ensembl:ENSG00000237238 Ensembl:ENSG00000204564 Ensembl:ENSG00000167930 Ensembl:ENSG00000138180 Ensembl:ENSG00000153291 Ensembl:ENSG00000154319 Ensembl:ENSG00000162624 Ensembl:ENSG00000100142 Ensembl:ENSG00000188869 Ensembl:ENSG00000072840 Ensembl:ENSG00000213965 Ensembl:ENSG00000124486 Ensembl:ENSG00000168887";
        String ensemblIds = "ENSEMBL:ENSG00000095303	ENSEMBL:ENSG00000166783	ENSEMBL:ENSG00000004478	ENSEMBL:ENSG00000160310	ENSEMBL:ENSG00000135318	ENSEMBL:ENSG00000181214	ENSEMBL:ENSG00000112062	ENSEMBL:ENSG00000171357	ENSEMBL:ENSG00000206292	ENSEMBL:ENSG00000214452	ENSEMBL:ENSG00000232957	ENSEMBL:ENSG00000055950	ENSEMBL:ENSG00000187037	ENSEMBL:ENSG00000151364	ENSEMBL:ENSG00000168397	ENSEMBL:ENSG00000096070	ENSEMBL:ENSG00000006634	ENSEMBL:ENSG00000049192	ENSEMBL:ENSG00000156313	ENSEMBL:ENSG00000106211	ENSEMBL:ENSG00000156113	ENSEMBL:ENSG00000101331	ENSEMBL:ENSG00000161298	ENSEMBL:ENSG00000236821	ENSEMBL:ENSG00000086102	ENSEMBL:ENSG00000160679	ENSEMBL:ENSG00000187098	ENSEMBL:ENSG00000146828	ENSEMBL:ENSG00000198088	ENSEMBL:ENSG00000164099	ENSEMBL:ENSG00000131069	ENSEMBL:ENSG00000120093	ENSEMBL:ENSG00000182534	ENSEMBL:ENSG00000161203	ENSEMBL:ENSG00000128652	ENSEMBL:ENSG00000167279	ENSEMBL:ENSG00000011405	ENSEMBL:ENSG00000170579	ENSEMBL:ENSG00000131183	ENSEMBL:ENSG00000005469	ENSEMBL:ENSG00000204361	ENSEMBL:ENSG00000168827	ENSEMBL:ENSG00000121989	ENSEMBL:ENSG00000153002	ENSEMBL:ENSG00000117676	ENSEMBL:ENSG00000085552	ENSEMBL:ENSG00000189280	ENSEMBL:ENSG00000162585	ENSEMBL:ENSG00000105997	ENSEMBL:ENSG00000145248	ENSEMBL:ENSG00000154822	ENSEMBL:ENSG00000006740	ENSEMBL:ENSG00000129810	ENSEMBL:ENSG00000224295	ENSEMBL:ENSG00000175224	ENSEMBL:ENSG00000107537	ENSEMBL:ENSG00000095203	ENSEMBL:ENSG00000181090	ENSEMBL:ENSG00000151458	ENSEMBL:ENSG00000006534	ENSEMBL:ENSG00000115275	ENSEMBL:ENSG00000064687	ENSEMBL:ENSG00000149932	ENSEMBL:ENSG00000167700	ENSEMBL:ENSG00000105792	ENSEMBL:ENSG00000146039	ENSEMBL:ENSG00000080644	ENSEMBL:ENSG00000117114	ENSEMBL:ENSG00000135577	ENSEMBL:ENSG00000226712	ENSEMBL:ENSG00000231337	ENSEMBL:ENSG00000160325	ENSEMBL:ENSG00000197993	ENSEMBL:ENSG00000149970	ENSEMBL:ENSG00000171488	ENSEMBL:ENSG00000148942	ENSEMBL:ENSG00000198862	ENSEMBL:ENSG00000240771	ENSEMBL:ENSG00000234990	ENSEMBL:ENSG00000170144	ENSEMBL:ENSG00000047410	ENSEMBL:ENSG00000139155	ENSEMBL:ENSG00000180658	ENSEMBL:ENSG00000182224	ENSEMBL:ENSG00000231558	ENSEMBL:ENSG00000116731	ENSEMBL:ENSG00000168795	ENSEMBL:ENSG00000155893	ENSEMBL:ENSG00000179941	ENSEMBL:ENSG00000182858	ENSEMBL:ENSG00000177427	ENSEMBL:ENSG00000180263	ENSEMBL:ENSG00000104805	ENSEMBL:ENSG00000140876	ENSEMBL:ENSG00000203624	ENSEMBL:ENSG00000119655	ENSEMBL:ENSG00000151093	ENSEMBL:ENSG00000167494	ENSEMBL:ENSG00000119778	ENSEMBL:ENSG00000075539	ENSEMBL:ENSG00000089127	ENSEMBL:ENSG00000111859	ENSEMBL:ENSG00000183963	ENSEMBL:ENSG00000173226	ENSEMBL:ENSG00000139218	ENSEMBL:ENSG00000183808	ENSEMBL:ENSG00000115677	ENSEMBL:ENSG00000065183	ENSEMBL:ENSG00000116005	ENSEMBL:ENSG00000110723	ENSEMBL:ENSG00000123739	ENSEMBL:ENSG00000227420	ENSEMBL:ENSG00000177189	ENSEMBL:ENSG00000143149	ENSEMBL:ENSG00000163017	ENSEMBL:ENSG00000181323	ENSEMBL:ENSG00000105245	ENSEMBL:ENSG00000198807	ENSEMBL:ENSG00000116882	ENSEMBL:ENSG00000164484	ENSEMBL:ENSG00000068885	ENSEMBL:ENSG00000137841	ENSEMBL:ENSG00000073910	ENSEMBL:ENSG00000092203	ENSEMBL:ENSG00000128512	ENSEMBL:ENSG00000140274	ENSEMBL:ENSG00000018408	ENSEMBL:ENSG00000018625	ENSEMBL:ENSG00000154124	ENSEMBL:ENSG00000162191	ENSEMBL:ENSG00000231658	ENSEMBL:ENSG00000099977	ENSEMBL:ENSG00000228230	ENSEMBL:ENSG00000243896	ENSEMBL:ENSG00000183671	ENSEMBL:ENSG00000139357	ENSEMBL:ENSG00000072071	ENSEMBL:ENSG00000001631	ENSEMBL:ENSG00000104142	ENSEMBL:ENSG00000196843	ENSEMBL:ENSG00000100596	ENSEMBL:ENSG00000185838	ENSEMBL:ENSG00000101074	ENSEMBL:ENSG00000171100	ENSEMBL:ENSG00000072315	ENSEMBL:ENSG00000109066	ENSEMBL:ENSG00000131650	ENSEMBL:ENSG00000152402	ENSEMBL:ENSG00000172940	ENSEMBL:ENSG00000101473	ENSEMBL:ENSG00000172831	ENSEMBL:ENSG00000107147	ENSEMBL:ENSG00000213648	ENSEMBL:ENSG00000170166	ENSEMBL:ENSG00000124564	ENSEMBL:ENSG00000072133	ENSEMBL:ENSG00000185267	ENSEMBL:ENSG00000004766	ENSEMBL:ENSG00000111801	ENSEMBL:ENSG00000165802	ENSEMBL:ENSG00000160410	ENSEMBL:ENSG00000070961	ENSEMBL:ENSG00000048052	ENSEMBL:ENSG00000203908	ENSEMBL:ENSG00000188343	ENSEMBL:ENSG00000205015	ENSEMBL:ENSG00000221867	ENSEMBL:ENSG00000233927	ENSEMBL:ENSG00000174502	ENSEMBL:ENSG00000135838	ENSEMBL:ENSG00000140564	ENSEMBL:ENSG00000165801	ENSEMBL:ENSG00000160213	ENSEMBL:ENSG00000091140	ENSEMBL:ENSG00000164199	ENSEMBL:ENSG00000153551	ENSEMBL:ENSG00000130037	ENSEMBL:ENSG00000157873	ENSEMBL:ENSG00000091129	ENSEMBL:ENSG00000226111	ENSEMBL:ENSG00000185614	ENSEMBL:ENSG00000074842	ENSEMBL:ENSG00000099953	ENSEMBL:ENSG00000196220	ENSEMBL:ENSG00000136828	ENSEMBL:ENSG00000133424	ENSEMBL:ENSG00000224845	ENSEMBL:ENSG00000116171	ENSEMBL:ENSG00000182646	ENSEMBL:ENSG00000115947	ENSEMBL:ENSG00000162775	ENSEMBL:ENSG00000150681	ENSEMBL:ENSG00000214733	ENSEMBL:ENSG00000171503	ENSEMBL:ENSG00000105856	ENSEMBL:ENSG00000119326	ENSEMBL:ENSG00000132300	ENSEMBL:ENSG00000196576	ENSEMBL:ENSG00000165092	ENSEMBL:ENSG00000147912	ENSEMBL:ENSG00000162946	ENSEMBL:ENSG00000001626	ENSEMBL:ENSG00000197106	ENSEMBL:ENSG00000168026	ENSEMBL:ENSG00000237258	ENSEMBL:ENSG00000130052	ENSEMBL:ENSG00000137878	ENSEMBL:ENSG00000128567	ENSEMBL:ENSG00000115306	ENSEMBL:ENSG00000230141	ENSEMBL:ENSG00000160058	ENSEMBL:ENSG00000163630	ENSEMBL:ENSG00000241837	ENSEMBL:ENSG00000171044	ENSEMBL:ENSG00000150347	ENSEMBL:ENSG00000170385	ENSEMBL:ENSG00000033867	ENSEMBL:ENSG00000237787	ENSEMBL:ENSG00000040199	ENSEMBL:ENSG00000233813	ENSEMBL:ENSG00000020256	ENSEMBL:ENSG00000123447	ENSEMBL:ENSG00000141314	ENSEMBL:ENSG00000117859	ENSEMBL:ENSG00000215811	ENSEMBL:ENSG00000169006	ENSEMBL:ENSG00000170871	ENSEMBL:ENSG00000213923	ENSEMBL:ENSG00000117335	ENSEMBL:ENSG00000186790	ENSEMBL:ENSG00000113263	ENSEMBL:ENSG00000157388	ENSEMBL:ENSG00000111913	ENSEMBL:ENSG00000222040	ENSEMBL:ENSG00000163344	ENSEMBL:ENSG00000176371	ENSEMBL:ENSG00000100083	ENSEMBL:ENSG00000186451	ENSEMBL:ENSG00000137171	ENSEMBL:ENSG00000119685	ENSEMBL:ENSG00000196277	ENSEMBL:ENSG00000157734	ENSEMBL:ENSG00000084453	ENSEMBL:ENSG00000198753	ENSEMBL:ENSG00000146350	ENSEMBL:ENSG00000128656	ENSEMBL:ENSG00000145050	ENSEMBL:ENSG00000182040	ENSEMBL:ENSG00000121644	ENSEMBL:ENSG00000224383	ENSEMBL:ENSG00000223775	ENSEMBL:ENSG00000223898	ENSEMBL:ENSG00000088448	ENSEMBL:ENSG00000196502	ENSEMBL:ENSG00000104889	ENSEMBL:ENSG00000150275	ENSEMBL:ENSG00000215504	ENSEMBL:ENSG00000099992	ENSEMBL:ENSG00000188996	ENSEMBL:ENSG00000157017	ENSEMBL:ENSG00000155511	ENSEMBL:ENSG00000236840	ENSEMBL:ENSG00000130540	ENSEMBL:ENSG00000181171	ENSEMBL:ENSG00000184895	ENSEMBL:ENSG00000168309	ENSEMBL:ENSG00000159708	ENSEMBL:ENSG00000078795	ENSEMBL:ENSG00000196367	ENSEMBL:ENSG00000113810	ENSEMBL:ENSG00000147099	ENSEMBL:ENSG00000235744	ENSEMBL:ENSG00000076641	ENSEMBL:ENSG00000150471	ENSEMBL:ENSG00000163106	ENSEMBL:ENSG00000126947	ENSEMBL:ENSG00000213599	ENSEMBL:ENSG00000205359	ENSEMBL:ENSG00000173662	ENSEMBL:ENSG00000149016	ENSEMBL:ENSG00000232962	ENSEMBL:ENSG00000100867	ENSEMBL:ENSG00000140522	ENSEMBL:ENSG00000240240	ENSEMBL:ENSG00000066923	ENSEMBL:ENSG00000091536	ENSEMBL:ENSG00000174500	ENSEMBL:ENSG00000126759	ENSEMBL:ENSG00000065371	ENSEMBL:ENSG00000187908	ENSEMBL:ENSG00000164776	ENSEMBL:ENSG00000211684	ENSEMBL:ENSG00000185873	ENSEMBL:ENSG00000140937	ENSEMBL:ENSG00000152056	ENSEMBL:ENSG00000101180	ENSEMBL:ENSG00000144290	ENSEMBL:ENSG00000171723	ENSEMBL:ENSG00000204694	ENSEMBL:ENSG00000197165	ENSEMBL:ENSG00000133640	ENSEMBL:ENSG00000241935	ENSEMBL:ENSG00000107736	ENSEMBL:ENSG00000163092	ENSEMBL:ENSG00000159352	ENSEMBL:ENSG00000196449	ENSEMBL:ENSG00000186105	ENSEMBL:ENSG00000167306	ENSEMBL:ENSG00000078674	ENSEMBL:ENSG00000169891	ENSEMBL:ENSG00000133107	ENSEMBL:ENSG00000221917	ENSEMBL:ENSG00000152689	ENSEMBL:ENSG00000133961	ENSEMBL:ENSG00000174595	ENSEMBL:ENSG00000151692	ENSEMBL:ENSG00000130487	ENSEMBL:ENSG00000103111	ENSEMBL:ENSG00000138472	ENSEMBL:ENSG00000162813	ENSEMBL:ENSG00000203797	ENSEMBL:ENSG00000121057	ENSEMBL:ENSG00000161981	ENSEMBL:ENSG00000120798	ENSEMBL:ENSG00000196535	ENSEMBL:ENSG00000174672	ENSEMBL:ENSG00000100380	ENSEMBL:ENSG00000204568	ENSEMBL:ENSG00000105641	ENSEMBL:ENSG00000165240	ENSEMBL:ENSG00000180871	ENSEMBL:ENSG00000197863	ENSEMBL:ENSG00000107159	ENSEMBL:ENSG00000197208	ENSEMBL:ENSG00000182271	ENSEMBL:ENSG00000108840	ENSEMBL:ENSG00000132164	ENSEMBL:ENSG00000077274	ENSEMBL:ENSG00000010282	ENSEMBL:ENSG00000137672	ENSEMBL:ENSG00000157087	ENSEMBL:ENSG00000114631	ENSEMBL:ENSG00000171385	ENSEMBL:ENSG00000138381	ENSEMBL:ENSG00000105289	ENSEMBL:ENSG00000073756	ENSEMBL:ENSG00000197849	ENSEMBL:ENSG00000127824	ENSEMBL:ENSG00000176142	ENSEMBL:ENSG00000106541	ENSEMBL:ENSG00000168385	ENSEMBL:ENSG00000133460	ENSEMBL:ENSG00000074276	ENSEMBL:ENSG00000239730	ENSEMBL:ENSG00000173467	ENSEMBL:ENSG00000221838	ENSEMBL:ENSG00000080845	ENSEMBL:ENSG00000114861	ENSEMBL:ENSG00000142910	ENSEMBL:ENSG00000100448	ENSEMBL:ENSG00000147168	ENSEMBL:ENSG00000227894	ENSEMBL:ENSG00000198399	ENSEMBL:ENSG00000153495	ENSEMBL:ENSG00000204970	ENSEMBL:ENSG00000226321	ENSEMBL:ENSG00000152332	ENSEMBL:ENSG00000229861	ENSEMBL:ENSG00000111252	ENSEMBL:ENSG00000188991	ENSEMBL:ENSG00000204252	ENSEMBL:ENSG00000198353	ENSEMBL:ENSG00000115594	ENSEMBL:ENSG00000154928	ENSEMBL:ENSG00000234347	ENSEMBL:ENSG00000145536	ENSEMBL:ENSG00000176057	ENSEMBL:ENSG00000132466	ENSEMBL:ENSG00000100348	ENSEMBL:ENSG00000205758	ENSEMBL:ENSG00000204683	ENSEMBL:ENSG00000126705	ENSEMBL:ENSG00000089177	ENSEMBL:ENSG00000177428	ENSEMBL:ENSG00000102805	ENSEMBL:ENSG00000106078	ENSEMBL:ENSG00000163743	ENSEMBL:ENSG00000159346	ENSEMBL:ENSG00000079277	ENSEMBL:ENSG00000134242	ENSEMBL:ENSG00000112337	ENSEMBL:ENSG00000115705	ENSEMBL:ENSG00000215762	ENSEMBL:ENSG00000206517	ENSEMBL:ENSG00000104814	ENSEMBL:ENSG00000071242	ENSEMBL:ENSG00000102858	ENSEMBL:ENSG00000187961	ENSEMBL:ENSG00000164588	ENSEMBL:ENSG00000163349	ENSEMBL:ENSG00000160767	ENSEMBL:ENSG00000171462	ENSEMBL:ENSG00000183605	ENSEMBL:ENSG00000144567	ENSEMBL:ENSG00000183473	ENSEMBL:ENSG00000133019	ENSEMBL:ENSG00000100092	ENSEMBL:ENSG00000206472	ENSEMBL:ENSG00000011114	ENSEMBL:ENSG00000197172	ENSEMBL:ENSG00000125257	ENSEMBL:ENSG00000107929	ENSEMBL:ENSG00000124568	ENSEMBL:ENSG00000149488	ENSEMBL:ENSG00000165959	ENSEMBL:ENSG00000077498	ENSEMBL:ENSG00000128536	ENSEMBL:ENSG00000188130	ENSEMBL:ENSG00000134313	ENSEMBL:ENSG00000178252	ENSEMBL:ENSG00000104973	ENSEMBL:ENSG00000105409	ENSEMBL:ENSG00000151917	ENSEMBL:ENSG00000171444	ENSEMBL:ENSG00000198467	ENSEMBL:ENSG00000146477	ENSEMBL:ENSG00000163291	ENSEMBL:ENSG00000005483	ENSEMBL:ENSG00000136830	ENSEMBL:ENSG00000169554	ENSEMBL:ENSG00000168243	ENSEMBL:ENSG00000163399	ENSEMBL:ENSG00000109572	ENSEMBL:ENSG00000121542	ENSEMBL:ENSG00000132950	ENSEMBL:ENSG00000003147	ENSEMBL:ENSG00000144485	ENSEMBL:ENSG00000170956	ENSEMBL:ENSG00000115109	ENSEMBL:ENSG00000137501	ENSEMBL:ENSG00000214424	ENSEMBL:ENSG00000103653	ENSEMBL:ENSG00000173193	ENSEMBL:ENSG00000145882	ENSEMBL:ENSG00000225581	ENSEMBL:ENSG00000129292	ENSEMBL:ENSG00000127377	ENSEMBL:ENSG00000113119	ENSEMBL:ENSG00000107719	ENSEMBL:ENSG00000100147	ENSEMBL:ENSG00000160746	ENSEMBL:ENSG00000187155	ENSEMBL:ENSG00000167671	ENSEMBL:ENSG00000110931	ENSEMBL:ENSG00000164485	ENSEMBL:ENSG00000160209	ENSEMBL:ENSG00000155495	ENSEMBL:ENSG00000042781	ENSEMBL:ENSG00000173486	ENSEMBL:ENSG00000137693	ENSEMBL:ENSG00000188566	ENSEMBL:ENSG00000211640	ENSEMBL:ENSG00000159086	ENSEMBL:ENSG00000127990	ENSEMBL:ENSG00000064932	ENSEMBL:ENSG00000138074	ENSEMBL:ENSG00000162889	ENSEMBL:ENSG00000072609	ENSEMBL:ENSG00000167230	ENSEMBL:ENSG00000128242	ENSEMBL:ENSG00000205356	ENSEMBL:ENSG00000054654	ENSEMBL:ENSG00000172943	ENSEMBL:ENSG00000205785	ENSEMBL:ENSG00000213402	ENSEMBL:ENSG00000140157	ENSEMBL:ENSG00000214078	ENSEMBL:ENSG00000149582	ENSEMBL:ENSG00000168894	ENSEMBL:ENSG00000073150	ENSEMBL:ENSG00000135917	ENSEMBL:ENSG00000127191	ENSEMBL:ENSG00000206579	ENSEMBL:ENSG00000006831	ENSEMBL:ENSG00000135063	ENSEMBL:ENSG00000205726	ENSEMBL:ENSG00000161217	ENSEMBL:ENSG00000203879	ENSEMBL:ENSG00000164961	ENSEMBL:ENSG00000232289	ENSEMBL:ENSG00000162613	ENSEMBL:ENSG00000214689	ENSEMBL:ENSG00000215750	ENSEMBL:ENSG00000162687	ENSEMBL:ENSG00000129255	ENSEMBL:ENSG00000211681	ENSEMBL:ENSG00000178233	ENSEMBL:ENSG00000189320	ENSEMBL:ENSG00000129194	ENSEMBL:ENSG00000072201	ENSEMBL:ENSG00000157765	ENSEMBL:ENSG00000105668	ENSEMBL:ENSG00000146910	ENSEMBL:ENSG00000198010	ENSEMBL:ENSG00000196686	ENSEMBL:ENSG00000188186	ENSEMBL:ENSG00000158552	ENSEMBL:ENSG00000176787	ENSEMBL:ENSG00000127588	ENSEMBL:ENSG00000136541	ENSEMBL:ENSG00000182871	ENSEMBL:ENSG00000042062	ENSEMBL:ENSG00000091181	ENSEMBL:ENSG00000111181	ENSEMBL:ENSG00000091136	ENSEMBL:ENSG00000168065	ENSEMBL:ENSG00000105696	ENSEMBL:ENSG00000094914	ENSEMBL:ENSG00000198223	ENSEMBL:ENSG00000124207	ENSEMBL:ENSG00000137707	ENSEMBL:ENSG00000196660	ENSEMBL:ENSG00000228562	ENSEMBL:ENSG00000228743	ENSEMBL:ENSG00000138483	ENSEMBL:ENSG00000233749	ENSEMBL:ENSG00000184408	ENSEMBL:ENSG00000242711	ENSEMBL:ENSG00000227328	ENSEMBL:ENSG00000140563	ENSEMBL:ENSG00000236282	ENSEMBL:ENSG00000150403	ENSEMBL:ENSG00000095752	ENSEMBL:ENSG00000058866	ENSEMBL:ENSG00000156535	ENSEMBL:ENSG00000168418	ENSEMBL:ENSG00000150093	ENSEMBL:ENSG00000116209	ENSEMBL:ENSG00000240118	ENSEMBL:ENSG00000153234	ENSEMBL:ENSG00000178171	ENSEMBL:ENSG00000214405	ENSEMBL:ENSG00000215251	ENSEMBL:ENSG00000188474	ENSEMBL:ENSG00000140750	ENSEMBL:ENSG00000007171	ENSEMBL:ENSG00000122679	ENSEMBL:ENSG00000243335	ENSEMBL:ENSG00000165388	ENSEMBL:ENSG00000121552	ENSEMBL:ENSG00000079691	ENSEMBL:ENSG00000197241	ENSEMBL:ENSG00000230780	ENSEMBL:ENSG00000058063	ENSEMBL:ENSG00000197683	ENSEMBL:ENSG00000151687	ENSEMBL:ENSG00000234357	ENSEMBL:ENSG00000136286	ENSEMBL:ENSG00000205636	ENSEMBL:ENSG00000174899	ENSEMBL:ENSG00000230518	ENSEMBL:ENSG00000148218	ENSEMBL:ENSG00000107960	ENSEMBL:ENSG00000124205	ENSEMBL:ENSG00000164050	ENSEMBL:ENSG00000180708	ENSEMBL:ENSG00000143476	ENSEMBL:ENSG00000166013	ENSEMBL:ENSG00000102606	ENSEMBL:ENSG00000119686	ENSEMBL:ENSG00000070019	ENSEMBL:ENSG00000004866	ENSEMBL:ENSG00000163645	ENSEMBL:ENSG00000072041	ENSEMBL:ENSG00000116544	ENSEMBL:ENSG00000070601	ENSEMBL:ENSG00000151532	ENSEMBL:ENSG00000183337	ENSEMBL:ENSG00000132746	ENSEMBL:ENSG00000225534	ENSEMBL:ENSG00000067842	ENSEMBL:ENSG00000089639	ENSEMBL:ENSG00000168090	ENSEMBL:ENSG00000188707	ENSEMBL:ENSG00000115685	ENSEMBL:ENSG00000119865	ENSEMBL:ENSG00000159873	ENSEMBL:ENSG00000167491	ENSEMBL:ENSG00000126216	ENSEMBL:ENSG00000080493	ENSEMBL:ENSG00000109118	ENSEMBL:ENSG00000134538	ENSEMBL:ENSG00000154175	ENSEMBL:ENSG00000182010	ENSEMBL:ENSG00000160460	ENSEMBL:ENSG00000099910	ENSEMBL:ENSG00000142279	ENSEMBL:ENSG00000226588	ENSEMBL:ENSG00000225697	ENSEMBL:ENSG00000148110	ENSEMBL:ENSG00000206199	ENSEMBL:ENSG00000042813	ENSEMBL:ENSG00000197415	ENSEMBL:ENSG00000115204	ENSEMBL:ENSG00000163389	ENSEMBL:ENSG00000092140	ENSEMBL:ENSG00000140297	ENSEMBL:ENSG00000091138	ENSEMBL:ENSG00000116039	ENSEMBL:ENSG00000142327";
        Bingo bingo = new Bingo(Constants.HUMAN,bingoOutDir);
        bingo.analyze(ensemblIds,"PDZK1-1");
        return;

        */
        try
        {
            BufferedWriter bw = new BufferedWriter(new FileWriter(new File(bingoOutDir +"/BingoTables.txt")));
            makeBingoTablesTxt(bw, Constants.HUMAN);
            makeBingoTablesTxt(bw, Constants.WORM);
            makeBingoTablesTxt(bw, Constants.FLY);
            bw.close();

        }
        catch(Exception  e)
        {
            System.out.println("Exception: " + e);
        }
    }
    private static List readBingoFileNames(String bingoOutFiles)
    {
        List fileNameList = new ArrayList();

        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(bingoOutFiles)));
            String line = "";
            while(true)
            {
                line=br.readLine();
                if (line== null) break;
                fileNameList.add(line);
            }
            br.close();

        }
        catch(Exception e)
        {
            System.out.println("Exception: "+e);
        }
        return fileNameList;

    }
    private static class BingoInfo
    {
        public String goID = "";
        public double corrPValue = 0.0;
        public String descr = "";
    }
    private static List getSignificantGO(String bingoFileName, double signif)
    {
        List sigBingoInfoList = new ArrayList();
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(bingoFileName)));
            String line = "";
            boolean start = false;
            while((line=br.readLine())!=null)
            {
                if (line.startsWith("GO-ID"))
                {
                    start= true;
                }
                if (!line.startsWith("GO-ID") && start)
                {
                    String[] splitLine =line.split("\t");
                    String goID = splitLine[0];
                    String corrPValueString = splitLine[2];
                    String descr = splitLine[7];
                    //System.out.println(goID + "\t" + corrPValueString + "\t" + descr);

                    double corrPValue = Double.parseDouble(corrPValueString);

                    if (corrPValue < signif)
                    {
                        BingoInfo bi = new BingoInfo();
                        bi.goID = goID;
                        bi.corrPValue = corrPValue;
                        bi.descr = descr;
                        sigBingoInfoList.add(bi);
                        System.out.println(goID + "\t" + corrPValue + "\t" + descr);

                    }
                }

            }
            br.close();
        }
        catch(Exception e)
        {
            System.out.println("Exception: "+e);
        }
        return sigBingoInfoList;

    }
    private static List getTopGOList(String bingoFileName, int numTop)
    {
        List sigBingoInfoList = new ArrayList();
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(bingoFileName)));
            String line = "";
            boolean start = false;
            int it = 0;
            while((line=br.readLine())!=null)
            {
                if (line.startsWith("GO-ID"))
                {
                    start= true;
                }
                if (!line.startsWith("GO-ID") && start)
                {
                    String[] splitLine =line.split("\t");
                    String goID = splitLine[0];
                    String corrPValueString = splitLine[2];
                    String descr = splitLine[7];
                    //System.out.println(goID + "\t" + corrPValueString + "\t" + descr);

                    double corrPValue = Double.parseDouble(corrPValueString);
                    if (it < numTop)
                    {
                        BingoInfo bi = new BingoInfo();
                        bi.goID = goID;
                        bi.corrPValue = corrPValue;
                        bi.descr = descr;
                        sigBingoInfoList.add(bi);
                        System.out.println(goID + "\t" + corrPValue + "\t" + descr);
                    }
                    else
                        break;

                    it = it+1;

                }

            }
            br.close();
        }
        catch(Exception e)
        {
            System.out.println("Exception: "+e);
        }
        return sigBingoInfoList;

    }
    public static void makeBingoTablesTxt(BufferedWriter bw, String organism)
    {

        String caption = "";
        String bingoFilesFileName = "";
        if (organism.equals(Constants.HUMAN))
        {
            caption = "Human Bingo Tables";
            bingoFilesFileName = "/Users/shirleyhui/Desktop/Results/Bingo/humanBingoFiles.txt";
        }
        else if (organism.equals(Constants.FLY))
        {
            caption = "Fly Bingo Tables";
            bingoFilesFileName = "/Users/shirleyhui/Desktop/Results/Bingo/flyBingoFiles.txt";
        }
        else
        {
            caption = "Worm Bingo Tables";
            bingoFilesFileName = "/Users/shirleyhui/Desktop/Results/Bingo/wormBingoFiles.txt";
        }


        List bingoFileNameList = readBingoFileNames(bingoFilesFileName);
        try
        {

            bw.write(caption +"\n");
            for(int i=0 ; i < bingoFileNameList.size();i++)
            {

                double signif = CUTOFF;

                String bingoFileName = (String)bingoFileNameList.get(i);
                int dotIx = bingoFileName.lastIndexOf(".");
                int slashIx = bingoFileName.lastIndexOf("/");

                String domainName = bingoFileName.substring(slashIx+1, dotIx);
                List sigBingoInfoList = getSignificantGO(bingoFileName, signif);

                String tableDomainName = "";
                if (sigBingoInfoList.size()==0)
                    tableDomainName = domainName +" *";
                else
                    tableDomainName = domainName;
                bw.write("\t\t\n"+tableDomainName+"\t\t\n");
                bw.write("GO ID\tp-value\tDescription\n");


                for (int j = 0; j < sigBingoInfoList.size();j++)
                {

                    BingoInfo info = (BingoInfo) sigBingoInfoList.get(j);

                    DecimalFormat dm = new DecimalFormat("0.###E0");
                    bw.write(info.goID +"\t");
                    bw.write(dm.format(info.corrPValue)+"\t");
                    bw.write(info.descr+"\n");

                }

                if (sigBingoInfoList.size()==0)
                {

                    sigBingoInfoList = getTopGOList(bingoFileName, 10);
                    for (int j = 0; j < sigBingoInfoList.size();j++)
                    {
                        BingoInfo info = (BingoInfo) sigBingoInfoList.get(j);

                        if (info.goID.equals("-1"))
                        {
                            bw.write("\t\tNone\n");
                        }
                        else
                        {
                            DecimalFormat dm = new DecimalFormat("0.###E0");
                            bw.write(info.goID +"\t");
                            bw.write(dm.format(info.corrPValue)+"\t");
                            bw.write(info.descr+"\n");
                        }

                    }
                }

            }

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();
        }

    }
}
