package org.baderlab.pdzsvm.predictor.mdsm;

import org.baderlab.pdzsvm.predictor.Predictor;
import org.baderlab.pdzsvm.predictor.nn.NN;

import java.util.*;
import java.io.*;

import org.baderlab.pdzsvm.validation.ValidationParameters;
import org.baderlab.pdzsvm.data.DataLoader;
import org.baderlab.pdzsvm.data.Data;
import org.baderlab.pdzsvm.data.manager.DataFileManager;
import org.baderlab.brain.ProteinProfile;
import org.biojava.bio.seq.Sequence;
import org.baderlab.pdzsvm.evaluation.Prediction;
import org.baderlab.pdzsvm.evaluation.Evaluation;
import weka.core.Instances;
import org.baderlab.pdzsvm.utils.Constants;
import org.baderlab.pdzsvm.utils.PDZSVMUtils;

/**
 * Copyright (c) 2010 University of Toronto
 * Code written by: Shirley Hui
 * Authors: Shirley Hui, Gary Bader
 *
 * This file is part of PDZSVM.
 *
 * PDZSVM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PDZSVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The software and
 * documentation provided hereunder is on an "as is" basis, and the
 * University of Toronto has no obligations to provide maintenance,
 * support, updates, enhancements or modifications.  In no event shall
 * the University of Toronto be liable to any party for direct, indirect,
 * special, incidental or consequential damages, including lost profits,
 * arising out of the use of this software and its documentation, even if
 * the University of Toronto has been advised of the possibility of such
 * damage. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PDZSVM.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Stiffler's Multi Domain Specificity Model predictor.
 * This predictor does not do any training on data, it sums the appropriate
 * weights values according to published weights for modelled domains.
 * Stiffler, M. et al. (2007) PDZ domain binding selectivity is optimized
 *   across the mouse proteome, Science, 317, 364-369.
 */
public class MDSMPredictor extends Predictor
{
    private final static String alphabet = "GAVLIMPFWSTNQYCKRHDE";
    private HashMap domainToModelMap = new HashMap();
    private int NUM_POS = 5;
    private int NUM_RES = alphabet.length();
    private int M = 5;
    private class MDSMModel
    {
        String domainName;
        double cutoff;
        double[][] wm;
        String domainSeq;

        public double score(String peptide)
        {
            double score = 0.0;
            for (int i=0; i < peptide.length();i++)
            {
                char residue = peptide.charAt(i);

                int resIndex = alphabet.indexOf(residue);
                double weight = 0.0;
                if (resIndex > -1)
                {
                    weight = wm[resIndex][i];
                }

                score = score + weight;
            }

            return score;
        }
    }

    public MDSMPredictor(List posTrainProfileList, List negTrainProfileList)
    {

        super(posTrainProfileList, negTrainProfileList);

        predictorName = "MDSM";

    }
    private double[][] initWeightMatrix(int r, int c)
    {
        double[][] weightMatrix= new double[r][c];
        for (int i=0; i < r;i++)
        {
            for (int j = 0; j < c;j++)
            {
                weightMatrix[i][j] = 0.0;
            }
        }
        return weightMatrix;
    }
    private void print(double[][] m )
    {
        for (int i=0; i < m.length;i++)
        {
            for (int j = 0; j < m[i].length;j++)
            {
                System.out.print(m[i][j]+"\t");
            }
            System.out.println();
        }
    }
    public void train()
    {
        //Load parameters
        HashMap domainNameToCutoffMap = new HashMap();
        String fileName = DataFileManager.DATA_ROOT_DIR+"/Data/Stiffler/Model/Stiffler_MDSM_Model_Cutoffs.txt";
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(fileName)));
            String line = "";
            while((line=br.readLine())!= null)
            {
                String[] splitString = line.split("\\s");
                String domainName = splitString[0];
                int ix = 1;
                if (M==10)
                {
                    ix = 2;
                }
                else if (M==15)
                {
                    ix = 3;
                }
                double cutoff = Double.parseDouble(splitString[ix]);

                domainNameToCutoffMap.put(domainName, cutoff);

            }

        }
        catch(Exception e)
        {
            System.out.println("Exception: " + e);
            e.printStackTrace();

        }


        fileName = DataFileManager.DATA_ROOT_DIR+"/Data/Stiffler/Model/Stiffler_MDSM_Model.txt";
        try
        {
            BufferedReader br = new BufferedReader(new FileReader(new File(fileName)));
            String line = "";
            while((line=br.readLine())!= null)
            {
                String[] splitString = line.split("\\s");
                String domainName = splitString[0];
                double[][] wm = initWeightMatrix(NUM_RES, NUM_POS);

                int numPos= 0;
                int numRes = 0;
                for (int i=1; i < splitString.length;i++)
                {
                    double weight = Double.parseDouble(splitString[i]);
                    wm[numRes][numPos] = weight;
                    numRes = numRes+1;
                    if (numRes == 20)
                    {
                        //System.out.println(numRes + "\t" + numPos + "\t" + i +"\t" + splitString.length);

                        numRes = 0;
                        numPos = numPos +1;
                    }
                }

                MDSMModel m = new MDSMModel();
                m.domainName = domainName;
                //System.out.println(domainName);
                m.cutoff = (Double)domainNameToCutoffMap.get(domainName);
                m.wm = wm;

                domainToModelMap.put(domainName, m);
                //System.out.println(domainName);
                // print(wm );
            }
            br.close();
        }
        catch(Exception e)
        {
            System.out.println("Exception1: " + e);
            e.printStackTrace();

        }
    }

    public static void main(String[] args)
    {
        DataLoader dl = new DataLoader();
        dl.loadMouseTrain();
        List posTrainProfileList = dl.getPosTrainProfileList();
        List negTrainProfileList = dl.getNegTrainProfileList();

        MDSMPredictor mdsm = new MDSMPredictor(posTrainProfileList, negTrainProfileList);
        mdsm.train();

        String testName = "";
        String dirName = "";
        //dl.loadMouseTest("ORPHAN"); testName = "MOUSE ORPHAN PM"; dirName = "PMMouseOrphanG";
        dl.loadWormTest(Constants.PROTEIN_MICROARRAY); testName = "PM WORM G";  dirName = "PMWormG";
        //dl.loadFlyTest(); testName = "FLY PM";   dirName = "PMFlyG";

        List posTestProfileList = dl.getPosTestProfileList();
        List negTestProfileList = dl.getNegTestProfileList();
        List predictionList = mdsm.predict(posTestProfileList, negTestProfileList);

        String predictorName = "MDSM";
        Evaluation eval = new Evaluation(predictionList);
        List rocAUCList = new ArrayList();
        rocAUCList.add(eval.getROCAUC());
        List prAUCList = new ArrayList();
        prAUCList.add(eval.getPRAUC());
        List aucLabelList = new ArrayList();
        aucLabelList.add(predictorName);
        Instances inst = eval.getCurve(1);
        List instList = new ArrayList();
        instList.add(inst);
        plotCurves(instList,rocAUCList, prAUCList, aucLabelList,predictorName + " (" +testName+")");

    }

    public List predict(List posTestProfileList, List negTestProfileList)
    {
        // Clear prediction list!
        predictionList = new ArrayList();
        Data testData = new Data();
        testData.addRawData(posTestProfileList, Constants.CLASS_YES);
        List balPosProfileList = new ArrayList();
        HashMap testPosProfileHashMap = PDZSVMUtils.profileListToHashMap(posTestProfileList);
        List balNegProfileList = new ArrayList();
        HashMap testNegProfileHashMap = new HashMap();
        if (negTestProfileList !=  null && !negTestProfileList.isEmpty())
        {
            testData.addRawData(negTestProfileList, Constants.CLASS_NO);
            testNegProfileHashMap = PDZSVMUtils.profileListToHashMap(negTestProfileList);
        }
        for (int i =0; i < negTestProfileList.size();i++)
        {
            ProteinProfile negProfile = (ProteinProfile)negTestProfileList.get(i);
            ProteinProfile posProfile = (ProteinProfile)testPosProfileHashMap.get(negProfile.getName());
            if (posProfile == null)
            {
                balNegProfileList.add(negProfile);
                balPosProfileList.add(null);
            }
            else
            {
                balNegProfileList.add(negProfile);
                balPosProfileList.add(posProfile);
            }
        }
        for (int i =0; i < posTestProfileList.size();i++)
        {
            ProteinProfile posProfile = (ProteinProfile)posTestProfileList.get(i);
            ProteinProfile negProfile = (ProteinProfile)testNegProfileHashMap.get(posProfile.getName());
            if (negProfile == null)
            {
                balNegProfileList.add(null);
                balPosProfileList.add(posProfile);
            }

        }
        ProteinProfile profile;
        for (int i =0;i < balPosProfileList.size();i++)
        {
            ProteinProfile testPosProfile = (ProteinProfile)balPosProfileList.get(i);
            ProteinProfile testNegProfile = (ProteinProfile)balNegProfileList.get(i);
            if (testPosProfile !=null)
                profile = testPosProfile;
            else
                profile = testNegProfile;
            String name = profile.getName();
            String organismLong = profile.getOrganism();
            String organism = PDZSVMUtils.organismLongToShortForm(organismLong);
            String methodLong = profile.getExperimentalMethod();
            String method = PDZSVMUtils.methodLongToShortForm(methodLong);
            String domainSeqFull = profile.getDomainSequence();

            ProteinProfile predictorProfile = NN.getNNBindingSiteSeqProfile(domainSeqFull, organism, posTrainProfileList);
            System.out.println("\tUsing: " + predictorProfile.getName());

            MDSMModel model = (MDSMModel) domainToModelMap.get(predictorProfile.getName());
            double cutoff = model.cutoff;

            if (testPosProfile!=null)
            {
                Collection testSeqCollection = testPosProfile.getSequenceMap();

                Iterator it = testSeqCollection.iterator();
                while(it.hasNext())
                {
                    Sequence seq = (Sequence)it.next();
                    String seqString = seq.seqString();

                    double mScore = model.score(seqString);
                    double score = mScore-cutoff;
                    Prediction prediction;

                    if (score > 0)
                    {
                        prediction = new Prediction(1.0, 1, score, name,domainSeqFull, seqString, organism, method );
                    }
                    else
                    {
                        prediction = new Prediction(0.0, 1, score, name,domainSeqFull, seqString, organism, method );


                    }
                    predictionList.add(prediction);

                }
            }
            if (testNegProfile!=null)
            {
                Collection testSeqCollection = testNegProfile.getSequenceMap();
                Iterator it = testSeqCollection.iterator();
                while(it.hasNext())
                {
                    Prediction prediction = null;
                    Sequence seq = (Sequence)it.next();
                    String seqString = seq.seqString();

                    double mScore = model.score(seqString);
                    double score = mScore-cutoff;

                    if (score <= 0)
                    {
                        prediction = new Prediction(0.0, 0, score, name,domainSeqFull, seqString, organism, method );
                    }
                    else
                    {
                        prediction = new Prediction(1.0, 0, score, name,domainSeqFull, seqString, organism, method );
                    }
                    predictionList.add(prediction);

                }
            }

        }
        if (negTestProfileList!=null && !negTestProfileList.isEmpty())
        {

            Evaluation eval = new Evaluation(predictionList);
            double rocAUC = eval.getROCAUC();
            if (rocAUC<0.5)
            {
                System.out.println("\tFlipping signs...");

                for (int ii = 0; ii < predictionList.size();ii++)
                {
                    Prediction prediction = (Prediction)predictionList.get(ii);
                    double decValue = prediction.getDecValue();
                    prediction.setDecValue(-decValue);
                }
            }
        }
        
        return predictionList;
    }

    public HashMap kFoldCrossValidation(ValidationParameters validParams)
    {
        return new HashMap();
    }

    public HashMap leaveOutCrossValidation(ValidationParameters validParams)
    {
        return new HashMap();
    }


}
