package org.pathwaycommons.trans;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.TreeMap;

import org.jdom.Namespace;
import org.jdom.Element;
import org.jdom.JDOMException;

import org.genemania.dw.entity.ExtResource;
import org.genemania.dw.tools.IdentifierMapperService;
import org.genemania.dw.util.DWUtil;
import org.genemania.dw.util.DefParams;
import org.genemania.dw.util.GenUtil;
import org.pathwaycommons.trans.util.XMLElementInfo;
import org.pathwaycommons.trans.util.XMLUtil;

/**
 * BIND to PSI-MI 2.5 xml translator. 
 * Source file names are presumed to be in the format: 'taxid9606.1.*'
 * mapXref () currently serves GI & EntrezID only. Assumes the source ID is
 * the primary ref, and the only new ref supported is Uniprot. Assumes GI is
 * the primary ref for BIND.
 *
 * @author rashadbadrawi
 */

public class BINDToPSIMITrans extends PSIMITrans {

    //local atts, reflecting build preferences
    private boolean proteinAFlag;       
    private boolean proteinBFlag;
    private int cntProteinA;
    private int cntProteinB;
    private boolean transComplexesFlag;
    private boolean mapIDsFlag;
    private boolean transProOnlyFlag;
    private boolean cleanDataFlag;

    private int cntTransInteraction;
    private int cntNonTransInteraction;
    private int cntTotalInteraction;
    private int cntBadInteraction;
    private String BINDInteractionID;                   //easier handling
    //atts used down the hierarchy
    protected static TreeMap  <String, String> interactionsMap = new TreeMap <String, String> ();
    protected static TreeMap <String, String> badInteractionsMap = new TreeMap <String, String> ();
    protected ArrayList <String> BINDSpeciesList = new ArrayList <String> ();
    protected String fileName;
    protected BufferedWriter bw;
    protected  String inputFilesPath;
    protected Namespace psins;
    protected int partID = 1;
    protected int interactionID = 1;
    protected int interactorID = 1;
    protected int expID = 1;
    protected int currentTaxID;
    protected boolean badInteractionData = false;

    protected BINDToPSIMITrans () {}
    
    protected BINDToPSIMITrans (String inputFilesPath) throws Exception {

        log = GenUtil.getDefaultLog();
        setInputPath (inputFilesPath);
        setMapIDsFlag (new Boolean (
          DefParams.getDefaultProp(DefParams.BIND_MAP_ID_PROP)).booleanValue());
        setTransComplexes (new Boolean (
          DefParams.getDefaultProp(DefParams.BIND_TRANS_COM_PROP)).booleanValue());
        setTransProOnly (new Boolean (
          DefParams.getDefaultProp(DefParams.BIND_TRANS_PRO_PROP)).booleanValue());
        setCleanData (new Boolean (
          DefParams.getDefaultProp(DefParams.BIND_CLEAN_DATA_PROP)).booleanValue());
        //global setters
        XMLElementInfo.readFromFile();
        CVMapEntry.loadCVMappings(
                  DefParams.getDefaultProp (DefParams.BIND_PSIMI_CV_FILE_PROP),
                  PSIMITags.BIND_SL, PSIMITags.DB_PSIMI);
    }

    private void setInputPath (String inputFilesPath) {

        this.inputFilesPath = inputFilesPath;
    }

    private void setMapIDsFlag (boolean mapIDsFlag) {

        this.mapIDsFlag = mapIDsFlag;
    }

    private void setTransComplexes (boolean transComplexes) {

        this.transComplexesFlag = transComplexes;
    }

    private void setTransProOnly (boolean transProOnly) {

        this.transProOnlyFlag = transProOnly;
    }

    private void setCleanData (boolean cleanDataFlag) {

        this.cleanDataFlag = cleanDataFlag;
    }

    public String getInputFilesPath () { return inputFilesPath; }

    public boolean isTransComplexes () { return this.transComplexesFlag; }

    public boolean isMapIDs () { return this.mapIDsFlag; }

    public boolean isTransProsOnly () { return this.transProOnlyFlag; }

    public boolean isCleanData () { return this.cleanDataFlag; }
    
    public void translate () throws Exception  {

        System.out.println ("Translation type: " + PSIMITrans.BIND_TO_PSIMI25);
        System.out.println ("Translating input: " + getInputFilesPath ());
        File sourceFileOrDir = new File (getInputFilesPath());
        if (!isMapIDs()) {
            System.out.println ("ID mapping service off.");
        }
        File badDataFile = new File (
             DefParams.getDefaultProp(DefParams.TOOLS_PATH_PROP), BAD_ENTRIES);
        bw = new BufferedWriter (new FileWriter (badDataFile));
        String outputFileName;
        if (sourceFileOrDir.isFile ()) {
            fileName = sourceFileOrDir.getAbsolutePath();
            this.currentTaxID = Integer.parseInt (fileName.substring(
                                fileName.indexOf(BINDTags.SOURCE_FILE_PREFIX) +
                                BINDTags.SOURCE_FILE_PREFIX.length (),
                                fileName.indexOf(GenUtil.DOT)));
            addSpecies (this.currentTaxID);
            if (isMapIDs()) {
                IdentifierMapperService.flush(
                                new String [] {ExtResource.LIST_UNIPROT},
                                new int [] {this.currentTaxID});
                IdentifierMapperService.init(
                        new String [] {ExtResource.LIST_UNIPROT},
                        new int [] {this.currentTaxID});
            }
            Element rootElement = XMLUtil.readSchemaBasedXML(fileName);
            Element PSIRootElement = translateBIND (null, rootElement);
            outputFileName = sourceFileOrDir.getName();
            //if (outputFileName.endsWith(XMLUtil.XML_FILE_SUFFIX)) {
            if (outputFileName.contains (XMLUtil.XML_FILE_SUFFIX)) {
                //hopefully, only one
                outputFileName = outputFileName.replace (XMLUtil.XML_FILE_SUFFIX,
                        PSIMI_FILE_SUFFIX + XMLUtil.XML_FILE_SUFFIX);
            }
            outputFileName = new File ( new File (DefParams.getDefaultProp(
                             DefParams.TOOLS_PATH_PROP)).getAbsolutePath(),
                             outputFileName).getAbsolutePath();
            XMLUtil.dump (PSIRootElement, outputFileName);
            rootElement = null;                         //attempt cleaning up
            PSIRootElement = null;
            System.gc();
        } else {            //} else if (sourceFileOrDir.isDirectory () {
            File filesArr [] = sourceFileOrDir.listFiles();
            Arrays.sort (filesArr);
            Element PSIRootElement = null;
            for (int i = 0; i < filesArr.length; i++) {
                int cnt = 1;
                //if (!filesArr [i].getAbsolutePath().endsWith(XMLUtil.XML_FILE_SUFFIX)) {
                if (!filesArr [i].getAbsolutePath().contains(XMLUtil.XML_FILE_SUFFIX)) {
                    continue;
                }
                fileName = filesArr [i].getAbsolutePath();
                System.out.println ("Processing file: " + cnt++ + " " + fileName);
                Element rootElement = XMLUtil.readSchemaBasedXML(fileName);
                int newTaxID = Integer.parseInt (fileName.substring(
                                    fileName.indexOf(BINDTags.SOURCE_FILE_PREFIX) +
                                    BINDTags.SOURCE_FILE_PREFIX.length (),
                                    fileName.indexOf(GenUtil.DOT)));
                if (newTaxID != this.currentTaxID) {
                    if(isMapIDs()) {
                        IdentifierMapperService.flush(
                                  new String [] {ExtResource.LIST_UNIPROT},
                                  new int [] {this.currentTaxID});
                    }
                    if (PSIRootElement != null) {
                        outputFileName = new File (
                                   new File (DefParams.getDefaultProp(
                                   DefParams.TOOLS_PATH_PROP)).getAbsolutePath(),
                                   BINDTags.SOURCE_FILE_PREFIX + this.currentTaxID +
                                   PSIMI_FILE_SUFFIX + XMLUtil.XML_FILE_SUFFIX).getAbsolutePath();
                        XMLUtil.dump (PSIRootElement, outputFileName);
                        PSIRootElement = null;
                        outputFileName = null;
                        //temp - reducing overhead, but prohibits 'some'
                        //inter-species interaction referencing
                        interactionsMap.clear();
                        badInteractionsMap.clear();
                    }
                    this.currentTaxID = newTaxID;
                    System.gc();
                } 
                if (isMapIDs()) {
                    IdentifierMapperService.init(
                              new String [] {ExtResource.LIST_UNIPROT},
                              new int [] {this.currentTaxID});
                }
                PSIRootElement = translateBIND (PSIRootElement, rootElement);
                rootElement = null;                      //attempt cleaning up
                System.gc();
           }
           //to handle the last set
           if (PSIRootElement != null) {
               outputFileName = new File (sourceFileOrDir.getAbsolutePath(),
                          BINDTags.SOURCE_FILE_PREFIX + this.currentTaxID +
                          PSIMI_FILE_SUFFIX + XMLUtil.XML_FILE_SUFFIX).getAbsolutePath();
               XMLUtil.dump (PSIRootElement, outputFileName);
               PSIRootElement = null;
               System.gc();
           }
        }
        /*
        if (isMapIDs()) {
            IdentifierMapperService.dumpUnmatchedUni(ExtResource.LIST_GI);
            IdentifierMapperService.dumpUnmatchedUni(ExtResource.LIST_ID);
        }
        */
        this.bw.flush();
        this.bw.close ();
    }

    private Element translateBIND (Element entrySetE, Element BINDSubmitE)
                                   throws Exception {

        if (entrySetE == null) {
            entrySetE = setEntrySet ();
        }
        translateAllInteractions (entrySetE, BINDSubmitE);

        return entrySetE;
    }

    private Element setEntrySet () {

        this.psins = Namespace.getNamespace(PSIMITags.PSIMI_NS);
        Element entrySetE = new Element (PSIMITags.ENTRY_SET, this.psins);
        entrySetE.setAttribute(XMLUtil.SCHEMA_LOC_ATT, PSIMITags.SCHEMA_LOC,
                         Namespace.getNamespace(XMLUtil.XSI_NAME, XMLUtil.XSI));
               
        entrySetE.setAttribute(PSIMITags.LEVEL_ATT, PSIMITags.LEVEL);
        entrySetE.setAttribute (PSIMITags.VERSION_ATT, PSIMITags.VERSION);

        return entrySetE;
    }

    private Element setEntry () throws JDOMException {

        Element entryE = new Element (PSIMITags.ENTRY, this.psins);

        Element sourceE = new Element (PSIMITags.SOURCE, this.psins);
        setCVName (sourceE, PSIMITags.BIND_SL);

        Element bibRefE = new Element (PSIMITags.BIB, this.psins);
        CVMapEntry CVMapper = CVMapEntry.getMappedEntry (PSIMITags.PUBMED_SL);
        setXRef (bibRefE, PSIMITags.PREF, CVMapper.getPSIMITerm_SL(),
                 CVMapper.getPSIMITermID (), BINDTags.PUB_ID,
                 PSIMITags.getRefTypeAccn(PSIMITags.REFTYPE_REF), PSIMITags.REFTYPE_REF);
        sourceE.addContent (bibRefE);

        entryE.addContent(sourceE);

        return entryE;
    }

    protected void setNames (Element parent, String shortLabel, String fullName) {

        Element namesE = new Element (PSIMITags.NAMES, this.psins);
        if (shortLabel != null) {
            Element shortLabelE = new Element (PSIMITags.SHORT_LABEL, this.psins);
            shortLabelE.addContent (shortLabel);
            namesE.addContent (shortLabelE);
        } else {
            System.err.println ("No short label provided: " + parent.getName());
        }
        if (fullName != null && !fullName.equals (NO_VAL)) {
            Element fullNameE = new Element (PSIMITags.FULLNAME, this.psins);
            fullNameE.addContent (fullName);
            namesE.addContent (fullNameE);
        }
        if (namesE.getChildren().size() > 0) {
            parent.addContent (namesE);
        }
    }

    //potential helper method
    protected void setCVName (Element parent, String sourceTerm) {

        CVMapEntry CVMapped = CVMapEntry.getMappedEntry(sourceTerm);
        setNames (parent, CVMapped.getPSIMITerm_SL(), CVMapped.getPSIMITerm_FL());
    }

    //potential helper method
    protected void setCVRef (Element parent, String sourceTerm, String refType,
                           String refElementName, String refValue) {

        CVMapEntry CVMapped = CVMapEntry.getMappedEntry(sourceTerm);
        setXRef (parent, refElementName, CVMapped.getPSIMITerm_SL(),
                 CVMapped.getPSIMITermID (), refValue,
                 PSIMITags.getRefTypeAccn(refType), refType);
    }

    protected void setXRef (Element parent, String refElementName, String DBName, String
                          DBID) {

        setXRef (parent, refElementName, DBName, null, DBID, null, null);
    }

    protected void setXRef (Element parent, String refElementName, String DBName, String
                          DBAccn, String DBID, String refTypeAccn, String refType) {

        Element xrefE = parent.getChild (PSIMITags.XREF, this.psins);
        if (xrefE == null) {
            xrefE = new Element (PSIMITags.XREF, this.psins);
            parent.addContent (xrefE);
        }
        setRef (xrefE, refElementName, DBName, DBAccn, DBID, refTypeAccn, refType);
    }

    protected void setRef (Element parent, String name, String DBName, String
                         DBAccn, String DBID, String refTypeAccn, String refType) {

       //check for redundancies
       ArrayList <Element> tempList = new ArrayList <Element> ();
       tempList.addAll (parent.getChildren());
       boolean found = false;
       for (int i = 0; i < tempList.size (); i++) {
           if (tempList.get (i).getAttributeValue(PSIMITags.DB_ATT).equals (DBName) &&
               tempList.get (i).getAttributeValue(PSIMITags.ID_ATT).equals (DBID)) {
               found = true;
               break;
           }
       }
       if (!found) {
           Element refE = new Element (name, this.psins);
           refE.setAttribute(PSIMITags.DB_ATT, DBName);
           if (DBAccn != null) {
               refE.setAttribute(PSIMITags.DBACC_ATT, DBAccn);
           }
           refE.setAttribute (PSIMITags.ID_ATT, DBID);
           if (refTypeAccn != null) {
               refE.setAttribute (PSIMITags.REFTYPEAC_ATT, refTypeAccn);
           }
           if (refType != null) {
               refE.setAttribute (PSIMITags.REFTYPE_ATT, refType);
           }
           parent.addContent (refE);
       }
    }

    private void translateAllInteractions (Element entrySetE, Element BINDSubmitE)
                                       throws Exception {

        Element entryE = entrySetE.getChild(PSIMITags.ENTRY, psins);
        if (entryE == null) {
            entryE = setEntry();
            entrySetE.addContent (entryE);
        }

        Element interactionListE = entryE.getChild (PSIMITags.INT_LIST, this.psins);
        if (interactionListE == null) {
            interactionListE = new Element (PSIMITags.INT_LIST, this.psins);
            entryE.addContent (interactionListE);
        }
        ArrayList <Element> bindIntList = XMLUtil.getElement (BINDTags.INTERACTION,
                                           BINDSubmitE, 1);
        for (int i = 0; i < bindIntList.size (); i++) {
            translateInteraction (interactionListE, bindIntList.get (i));
        }
        System.out.println ("For file: " + getInputFilesPath() + ": ");
        System.out.println ("Total Interactions: " + this.cntTotalInteraction +
                            " Translated: " + this.cntTransInteraction +
                            " Skipped: " + this.cntNonTransInteraction +
                            " Bad: " + this.cntBadInteraction +
                            " Int A - Protein: " + this.cntProteinA +
                            " Int B - Protein: " + this.cntProteinB);
        if (isTransComplexes()) {
            if (isTransProsOnly()) {
                System.err.println ("No support for complexes with 'protein only' option");
                return;
            }
            //translate complexes, as PSI-MI interactions
            BINDToPSIMITransComplex BINDComTrans =
                new BINDToPSIMITransComplex (inputFilesPath, fileName, psins,
                        interactionID, interactorID, partID, expID, bw);
            BINDComTrans.translateAllComplexes (entryE, BINDSubmitE);
        }
        dumpBINDSpecies();
    }

    //debugging
    private void listAllInteractions () {

        System.out.println ("Listing IDs for all interactions: BIND ID - PSI-MI ID");
        Iterator iterator = interactionsMap.keySet().iterator();
        while (iterator.hasNext ()) {
            String intID = (String)iterator.next ();
            System.out.println ("Int: " + intID + " " + interactionsMap.get (intID));
        }
        System.out.println ("Listing IDs for 'bad' interactions: BIND ID - PSI-MI ID");
        iterator = badInteractionsMap.keySet().iterator();
        while (iterator.hasNext ()) {
            String intID = (String)iterator.next ();
            System.out.println ("Bad Int: " + intID + " " + badInteractionsMap.get (intID));
        }
    }

    private void translateInteraction (Element interactionListE, 
                              Element BINDInteractionE) throws JDOMException, IOException {

        Element interactionE = new Element (PSIMITags.INT, this.psins);
        ArrayList <Element> eList = XMLUtil.getElement (BINDTags.INTERACTION_ID, BINDInteractionE,
                                    XMLUtil.getDepth (BINDTags.INTERACTION));
        BINDInteractionID = null;
        if (XMLUtil.validateSingle (BINDTags.INTERACTION_ID, eList) &&
            XMLUtil.validateInt(eList.get (0))) {
            BINDInteractionID = eList.get (0).getTextTrim ();
        }

        eList = XMLUtil.getElement (BINDTags.SIMPLE_DESC,
                   BINDInteractionE, XMLUtil.getDepth (BINDTags.INTERACTION));
        String shortLabel = NO_VAL;
        if (XMLUtil.validateSingle (BINDTags.SIMPLE_DESC, eList)) {
            shortLabel = getDefText (eList.get (0), NO_VAL);
        }
        if (!shortLabel.equals (NO_VAL)) {
            setNames (interactionE, shortLabel, shortLabel);
        } else {
            setNames (interactionE, shortLabel, null);
            this.badInteractionData = true;
        }
        setCVRef (interactionE, PSIMITags.BIND_SL, PSIMITags.REFTYPE_IDENT,
                  PSIMITags.PREF, BINDInteractionID);

        Element expListE = new Element (PSIMITags.EXP_LIST, this.psins);
        interactionE.addContent (expListE);
        translateExpDesc (expListE, BINDInteractionE, BINDInteractionID);
        Element partListE = new Element (PSIMITags.PARTLIST, this.psins);
        interactionE.addContent (partListE);

        //1st BIND interactor
        Element partE = new Element (PSIMITags.PART, this.psins);
        partE.setAttribute(PSIMITags.ID_ATT, Integer.toString (this.partID++));
        partListE.addContent (partE);
        Element interactorE = new Element (PSIMITags.INTERACTOR, this.psins);
        partE.addContent (interactorE);
        translateInteractor (interactorE, BINDInteractionE.getChild (BINDTags.INTERACT_A), this.interactorID++);

        //2nd BIND interactor
        partE = new Element (PSIMITags.PART, this.psins);
        partE.setAttribute(PSIMITags.ID_ATT, Integer.toString (this.partID++));
        partListE.addContent (partE);
        interactorE = new Element (PSIMITags.INTERACTOR, this.psins);
        partE.addContent (interactorE);
        translateInteractor (interactorE, BINDInteractionE.getChild(BINDTags.INTERACT_B), this.interactorID++);

        eList = XMLUtil.getElement (BINDTags.INTERACTION_COL, BINDInteractionE,
                                    XMLUtil.getDepth (BINDTags.INTERACTION));
        if (XMLUtil.validateSingle (BINDTags.INTERACTION_COL, eList)) {
            String BINDDivision = BINDTags.checkBINDDiv (eList.get (0).getTextTrim ());
            addAttribute (interactionE, PSIMITags.ATT_BIND_DIVISION,
                          BINDDivision);
        }
        addInteraction (interactionListE, interactionE);
    }

    //interaction counter increments regardless. Flags control.
    private void addInteraction (Element interactionListE, Element interactionE)
                                                          throws IOException {

        interactionsMap.put (BINDInteractionID, String.valueOf (interactionID));
        interactionE.setAttribute(PSIMITags.ID_ATT,
                                      Integer.toString (this.interactionID++));

        if (!this.transProOnlyFlag || (this.proteinAFlag && this.proteinBFlag)) {
            this.cntTransInteraction++;
            if (!this.badInteractionData) {
                interactionListE.addContent (interactionE);
            } else {
                if (!this.cleanDataFlag) {                //not cleaning anyway
                    interactionListE.addContent (interactionE);
                }
                this.badInteractionData = false;
                addAttribute (interactionE, PSIMITags.ATT_BAD_DATA_FILE, fileName);
                XMLUtil.dump(interactionE, bw);
                bw.newLine();
                this.cntBadInteraction++;
                badInteractionsMap.put (BINDInteractionID, 
                   interactionE.getAttributeValue (PSIMITags.ID_ATT));
            }
        } else {
            this.cntNonTransInteraction++;
            if (this.badInteractionData == true) {
                this.badInteractionData = false;
            }
        }
        this.cntTotalInteraction++;

        //track proteins anyway
        if (this.proteinAFlag) {
            this.cntProteinA++;
        }
        if (this.proteinBFlag) {
            this.cntProteinB++;
        }
        this.proteinAFlag = false;
        this.proteinBFlag = false;
    }

    private void translateExpDesc (Element expListE, Element BINDInteractionE, 
                                   String BINDInteractionID) throws JDOMException {

        ArrayList <Element> eList = XMLUtil.getElement (BINDTags.COND,
                         BINDInteractionE, XMLUtil.getDepth (BINDTags.INTERACTION));
        if (!XMLUtil.validateExists (BINDTags.COND, eList)) {     //no exp condition
            Element expDescE = new Element (PSIMITags.EXP_DESC, this.psins);
            expDescE.setAttribute(PSIMITags.ID_ATT, Integer.toString(this.expID++));
            expListE.addContent (expDescE);
            Element intDetMethodE = new Element (PSIMITags.INT_DET_METHOD, this.psins);
            setNames(intDetMethodE, PSIMITags.SL_METHOD_NS, PSIMITags.SL_METHOD_NS);
            setXRef(intDetMethodE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                   PSIMITags.ID_METHOD_NS, null, null);
            translatePubMed(expDescE,
                      BINDInteractionE.getChild (BINDTags.INTERACTION_SOURCE));
            if (expDescE.getChildren(PSIMITags.BIB, this.psins).size() == 0) {
                Element bibRefE = new Element (PSIMITags.BIB, this.psins);
                CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.BIND_SL);
                setXRef (bibRefE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                         CVMapped.getPSIMITermID(), BINDInteractionID, 
                         null, null);
                expDescE.addContent(bibRefE);
            }
            expDescE.addContent (intDetMethodE);
            //System.err.println ("No exp. conditions " + expID);
        } else {
            for (int i = 0; i < eList.size (); i++) {
                Element expDescE = new Element (PSIMITags.EXP_DESC, this.psins);
                expDescE.setAttribute(PSIMITags.ID_ATT, Integer.toString(this.expID++));
                expListE.addContent (expDescE);
                Element intDetMethodE = new Element (PSIMITags.INT_DET_METHOD, this.psins);
                translateDetectionMethod (expDescE, intDetMethodE, eList.get (i));
                if (intDetMethodE.getChildren().size() == 0) {
                    setNames(intDetMethodE, PSIMITags.SL_METHOD_NS, PSIMITags.SL_METHOD_NS);
                    setXRef(intDetMethodE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                            PSIMITags.ID_METHOD_NS, null, null);
                }
                translatePubMed (expDescE, eList.get (i)); //order is signifcant
                translatePubMed2 (expDescE, eList.get (i));
                if (expDescE.getChildren(PSIMITags.BIB, this.psins).size() == 0) {
                    Element bibRefE = new Element (PSIMITags.BIB, this.psins);
                    CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.BIND_SL);
                    setXRef (bibRefE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                         CVMapped.getPSIMITermID(), BINDInteractionID,
                         null, null);
                    expDescE.addContent(bibRefE);
                }
                expDescE.addContent (intDetMethodE);
            }
        }
    }

    private void translateDetectionMethod (Element expDescE, Element intDetMethodE,
                                  Element BINDConditionE) throws JDOMException {

        String shortLabel = null;
        String fullName = null;

        ArrayList <Element> tempList = XMLUtil.getElement (BINDTags.COND_DESC,
                BINDConditionE, XMLUtil.getDepth (BINDTags.COND));
        if (XMLUtil.validateSingle(BINDTags.COND_DESC, tempList)) {
            fullName = getDefText(tempList.get (0), NO_VAL);
        }
        tempList = XMLUtil.getElement (BINDTags.EXP_SYS, BINDConditionE,
                               XMLUtil.getDepth (BINDTags.COND));
        if (XMLUtil.validateSingle (BINDTags.EXP_SYS, tempList)) {
            shortLabel = tempList.get (0).getAttributeValue(BINDTags.VALUE_ATT);
            CVMapEntry CVMapped = CVMapEntry.getMappedEntry(shortLabel);
            //temp section, in case more BIND non-consistent exp. systems
            if (CVMapped == null) {
                System.err.println ("Error: Undetected BIND exp: " + shortLabel);
                setNames (intDetMethodE, shortLabel, shortLabel);
                setXRef(intDetMethodE, PSIMITags.PREF, PSIMITags.BIND_SL, PSIMITags.ID_BIND,
                        tempList.get (0).getTextTrim(), null, null);
            } else {
            //end temp section
                setNames (intDetMethodE, CVMapped.getPSIMITerm_SL(),
                          CVMapped.getPSIMITerm_FL());
                if (CVMapped.getPSIMITermID () != null) {
                    setXRef(intDetMethodE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                            CVMapped.getPSIMITermID (), null, null);
                } else {
                    setXRef(intDetMethodE, PSIMITags.PREF, PSIMITags.BIND_SL, PSIMITags.ID_BIND,
                            CVMapped.getSourceID(), null, null);
                }
            }
        }
        setNames(expDescE, shortLabel, fullName);
    }

    protected void translatePubMed (Element PSIReferencedE, Element
                                  BINDReferencedE) throws JDOMException {

        ArrayList <Element> pubMedList = XMLUtil.getElement (BINDTags.PUBMED_ID,
           BINDReferencedE, XMLUtil.getDepth (BINDReferencedE.getName()), true);
        translateReferences (pubMedList, PSIReferencedE);
    }
    
    protected void translatePubMed2 (Element PSIReferencedE, Element
                                  BINDReferencedE) throws JDOMException {

        ArrayList <Element> pubMedList = XMLUtil.getElement (BINDTags.PUBMEDMU_ID,
           BINDReferencedE, XMLUtil.getDepth (BINDReferencedE.getName()), true);
        translateReferences (pubMedList, PSIReferencedE);
    }
    
    //helper method
    protected void translateReferences (ArrayList <Element> pubMedList, Element
            PSIReferencedE) throws JDOMException {
        
        boolean addFlag = false;
        boolean firstTime = true;
        Element bibRefE = PSIReferencedE.getChild (PSIMITags.BIB, this.psins);
        if (bibRefE == null) {
            bibRefE = new Element (PSIMITags.BIB, this.psins);
            addFlag = true;
        } else {
            firstTime = false;
        }
        for (int j = 0; j < pubMedList.size (); j++) {
            CVMapEntry CVMapper = CVMapEntry.getMappedEntry (PSIMITags.PUBMED_SL);
            if (firstTime) {
                setXRef (bibRefE, PSIMITags.PREF, CVMapper.getPSIMITerm_SL(),
                        CVMapper.getPSIMITermID (),
                        getDefText(pubMedList.get (j), NO_VAL),
                        PSIMITags.getRefTypeAccn(PSIMITags.REFTYPE_REF),
                        PSIMITags.REFTYPE_REF);
                firstTime = false;
            } else {
                setXRef (bibRefE, PSIMITags.SREF, CVMapper.getPSIMITerm_SL(),
                        CVMapper.getPSIMITermID (),
                        getDefText(pubMedList.get (j), NO_VAL),
                        PSIMITags.getRefTypeAccn(PSIMITags.REFTYPE_REF),
                        PSIMITags.REFTYPE_REF);
            }
        }
        if (bibRefE.getChildren().size() > 0 && addFlag) {
            PSIReferencedE.addContent (bibRefE);
        }
    }

   protected void translateInteractor (Element interactorE, Element
                   BINDInteractorE, int interactorID) throws JDOMException {

        interactorE.setAttribute(PSIMITags.ID_ATT, Integer.toString (interactorID));

        Element interactorTypeE = new Element (PSIMITags.INT_TYPE, this.psins);
        Element BINDInteractorTypeE = translateInteractorType (interactorTypeE, BINDInteractorE);
        String shortLabel = NO_VAL;
        String fullName = null;
        Element BINDNameE = null;
        ArrayList <Element> tempList = XMLUtil.getElement (BINDTags.OBJ_SHORT_LABEL,
                        BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_SHORT_LABEL, tempList)) {
            shortLabel = getDefText(tempList.get (0), NO_VAL);
            BINDNameE = tempList.get (0);
        }
        if (shortLabel.equals (BINDToPSIMITrans.NO_VAL)) {
            this.badInteractionData = true;
        }
        tempList = XMLUtil.getElement (BINDTags.OBJ_DESC, BINDInteractorE,
                               XMLUtil.getDepth(BINDInteractorE.getName ()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_DESC, tempList)) {
            fullName = getDefText(tempList.get (0), NO_VAL);
        }
        //order of addition is significant.
        setNames (interactorE, shortLabel, fullName);
        tempList = XMLUtil.getElement (BINDTags.OBJ_EXTREF,
                BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        Element BINDExtRefE = null;
        if (XMLUtil.validateSingle(BINDTags.OBJ_EXTREF, tempList)) {
            BINDExtRefE = tempList.get (0);
        }
        //get organism info
        int interactorSpeciesID = -1;
        Element organismE = new Element (PSIMITags.ORG, this.psins);
        tempList = XMLUtil.getElement (BINDTags.OBJECT_ID_ID, BINDInteractorE,
                   XMLUtil.getDepth (BINDInteractorE.getName()), false);
        ArrayList <Element> orgNameList = XMLUtil.getElement (BINDTags.ORG_TAXNAME,
                      BINDInteractorE, XMLUtil.getDepth (BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJECT_ID_ID, tempList) &&
            XMLUtil.validateInt(tempList.get (0))) {
            organismE.setAttribute(PSIMITags.NCBI_TAXID,
                                   getDefText(tempList.get (0), NO_VAL));
            interactorSpeciesID = Integer.parseInt (getDefText(tempList.get (0), NO_VAL));

            if (XMLUtil.validateSingle (BINDTags.ORG_TAXNAME, orgNameList)) {
                String orgName = getDefText(orgNameList.get (0), NO_VAL);
                if (!orgName.equals (NO_VAL)) {
                    setNames (organismE, orgName, orgName);
                } else {
                    System.err.println ("Bad org name: " +
                                        orgNameList.get (0).getTextTrim());
                }
            } else {
                System.err.println ("Error: tax ID but no org name in BIND.");
            }
        } else {                                              //unidentified
            organismE.setAttribute (PSIMITags.NCBI_TAXID, PSIMITags.TAXID_UN_ID);
            setNames (organismE, PSIMITags.TAXID_UN_NAME, PSIMITags.TAXID_UN_NAME);
        }
        if (interactorSpeciesID == -1) {
            //System.err.println ("Missing org info for: " + interactorID);
            interactorSpeciesID = Integer.parseInt (PSIMITags.TAXID_UN_ID);
        }
        addSpecies(interactorSpeciesID);
        translateInteractorID (interactorE, BINDInteractorE, BINDExtRefE,
                 BINDNameE, BINDInteractorTypeE, interactorSpeciesID);    //forces null exception
        interactorE.addContent (interactorTypeE);
        //add organism now - for semantics
        if (organismE.getChildren ().size() > 0) {
            interactorE.addContent(organismE);
        }
    }

    private Element translateInteractorType (Element interactorTypeE, Element
                                    BINDInteractorE) throws JDOMException {

        //protein
        ArrayList <Element> tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_P,
                BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_P, tempList)) {
            CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_P);
            setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                       CVMapper.getPSIMITerm_FL());
            setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                    CVMapper.getTargetID(), null, null);
            //temp section to limit trans support for proteins
            if (BINDInteractorE.getName().equals(BINDTags.INTERACT_A)) {
                this.proteinAFlag = true;
            } else if (BINDInteractorE.getName().equals(BINDTags.INTERACT_B)) {
                this.proteinBFlag = true;
            }
            return tempList.get (0);
        }
        //Gene
        //skip for complexes, no such type for complexes in BIND
        if (!BINDInteractorE.getName().equals (BINDTags.COMPLEX_MOL_OBJ)) {
            tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_GENE,
                       BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
            if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_GENE, tempList)) {
                CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_GENE);
                setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                           CVMapper.getPSIMITerm_FL());
                setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                        CVMapper.getTargetID(), null, null);
                return tempList.get (0);
            }
        }
        //DNA
        tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_DNA,
                   BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_DNA, tempList)) {
            CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_DNA);
            setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                       CVMapper.getPSIMITerm_FL());
            setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                    CVMapper.getTargetID(), null, null);
            return tempList.get (0);
        }
        //RNA
        tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_RNA,
                   BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_RNA, tempList)) {
            CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_RNA);
            setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                       CVMapper.getPSIMITerm_FL());
            setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                    CVMapper.getTargetID(), null, null);
            return tempList.get (0);
        }
        //Complex
        tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_COM,
                   BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_COM, tempList)) {
            CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_COM);
            setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                       CVMapper.getPSIMITerm_FL());
            setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                    CVMapper.getTargetID(), null, null);
            return tempList.get (0);
        }
        //Small Molecule
        tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_SM,
                   BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
        if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_SM, tempList)) {
            CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_SM);
            setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                       CVMapper.getPSIMITerm_FL());
            setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                    CVMapper.getTargetID(), null, null);
            return tempList.get (0);
        }
        //not specified
        //skip for complexes, no such type for complexes in BIND
        if (!BINDInteractorE.getName().equals (BINDTags.COMPLEX_MOL_OBJ)) {
            tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_NS,
                       BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
            if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_NS, tempList)) {
                CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_NS);
                setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                           CVMapper.getPSIMITerm_FL());
                setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                        CVMapper.getTargetID(), null, null);
                return tempList.get (0);
            }
        }
        //Photon - no match in PSI-MI
        //skip for complexes, no such type for complexes in BIND
        if (!BINDInteractorE.getName().equals (BINDTags.COMPLEX_MOL_OBJ)) {
            tempList = XMLUtil.getElement (BINDTags.OBJ_TYPE_ID_PH,
                       BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
            if (XMLUtil.validateSingle(BINDTags.OBJ_TYPE_ID_PH, tempList)) {
                CVMapEntry CVMapper = CVMapEntry.getMappedEntry(BINDTags.OBJ_TYPE_ID_PH);
                setNames (interactorTypeE, CVMapper.getPSIMITerm_SL(),
                                           CVMapper.getPSIMITerm_FL());
                setXRef(interactorTypeE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.ID_PSIMI,
                        CVMapper.getTargetID(), null, null);
                //System.err.println ("No support for 'photon' type in PSI-MI.");
                return tempList.get (0);
            }
        }
        System.err.println ("Unsupported interactant type for: " +
                            BINDInteractorE.getName());
        return null;
    }

    private void translateInteractorID (Element interactorE, Element 
            BINDInteractorE, Element BINDExtRefE, Element BINDNameE, 
            Element BINDInteractorTypeE, int interactorSpeciesID) throws JDOMException {

        if (BINDNameE == null || BINDNameE.getText().trim().length () == 0 ||
            BINDNameE.getText().trim().equalsIgnoreCase(BINDTags.UNKNOWN) ||
            BINDNameE.getText().trim().equalsIgnoreCase(BINDTags.NULL_VAL)) {
            return;                                 //no dummy/default refElementName
        }
        //special complex related issue
        if (BINDInteractorTypeE == null) {
            System.err.println ("Missing interactor type: PSI-MI interactor ID: " +
                   interactorE.getAttributeValue (PSIMITags.ID_ATT) + " " + interactorSpeciesID);
            setXRef(interactorE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.TYPE_UP_ID);
            return;
        }
        String BINDInteractorType = BINDInteractorTypeE.getName();
        if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_NS) ||
            BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_PH)) {
            setXRef(interactorE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.TYPE_UP_ID);
            return;
        }
        boolean pRefSetFlag = false;
        ArrayList <Element> otherDBList;
        String interactorType = null, refType, refTypeAccn;
        if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_P) ||
            BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_DNA) || 
            BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_RNA) ||
            BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_GENE)) {
            if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_P)) {
                interactorType = PSIMITags.TYPE_PROTEIN_SL;
            } else if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_DNA)) {
                interactorType = PSIMITags.TYPE_DNA_SL;
            } else if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_RNA)) {
                interactorType = PSIMITags.TYPE_RNA_SL;
            } else if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_GENE)) {
                interactorType = PSIMITags.TYPE_GENE_SL;
            }
            ArrayList <Element> giList = new ArrayList <Element> (), 
                                domainIDList = new ArrayList <Element> ();
            //temp section
            Element BINDObjectID = BINDInteractorTypeE.getChild (BINDTags.BIND_ID);
            if (BINDObjectID.getChild(BINDTags.BIND_ID_GI) != null) {
                giList.addAll (BINDObjectID.getChild(BINDTags.BIND_ID_GI).
                               getChildren (BINDTags.ID_GI));
            }
            if (BINDObjectID.getChild(BINDTags.BIND_ID_DI) != null) {
                domainIDList.addAll (BINDObjectID.getChild(BINDTags.BIND_ID_DI).
                                     getChildren (BINDTags.DOMAIN_ID));
            }
            //end temp section
            if (XMLUtil.validateSingle(BINDTags.ID_GI, giList) &&
                XMLUtil.validateInt(giList.get (0))) {      //only one, if any
                mapXref (interactorE, interactorType, ExtResource.LIST_GI,
                         giList.get (0).getTextTrim (), interactorSpeciesID);
                pRefSetFlag = true;
            }

            if (XMLUtil.validateSingle(BINDTags.DOMAIN_ID, domainIDList) &&
                XMLUtil.validateInt(domainIDList.get (0))) {    //only one, if any
                CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.ENTREZ_GENE_DB);
                refType = CVMapped.getAddInfo(interactorType, domainIDList.get (0).getTextTrim());
                refTypeAccn = PSIMITags.getRefTypeAccn (refType);
                if (!pRefSetFlag) {
                    setXRef (interactorE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                             CVMapped.getPSIMITermID (),
                             domainIDList.get (0).getTextTrim(), refTypeAccn, refType);
                    mapXref (interactorE, interactorType, ExtResource.LIST_ID, 
                             domainIDList.get (0).getTextTrim(), interactorSpeciesID);
                    pRefSetFlag = true;
                } else {
                    setXRef (interactorE, PSIMITags.SREF, CVMapped.getPSIMITerm_SL(),
                             CVMapped.getPSIMITermID (),
                             domainIDList.get (0).getTextTrim(), refTypeAccn, refType);
                    mapXref (interactorE, interactorType, ExtResource.LIST_ID,
                             domainIDList.get (0).getTextTrim(), interactorSpeciesID);
                }
            }
        } else if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_SM)) {
            interactorType = PSIMITags.TYPE_SM_SL;
            ArrayList <Element> tempList = XMLUtil.getElement (BINDTags.SMALL_MOLECULE_ID,
                  BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
            if (XMLUtil.validateSingle(BINDTags.SMALL_MOLECULE_ID, tempList) &&
                XMLUtil.validateInt(tempList.get (0))) {    //only one, if any, internal ID
                CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.BIND_SL);
                //CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.BIND_SMID_SL);
                refType = CVMapped.getAddInfo(interactorType,
                                              tempList.get (0).getTextTrim ());
                refTypeAccn = PSIMITags.getRefTypeAccn(refType);
                setXRef (interactorE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                         CVMapped.getPSIMITermID (),
                         tempList.get (0).getTextTrim(), refTypeAccn, refType);
                pRefSetFlag = true;
            }
        } else if (BINDInteractorType.equals(BINDTags.OBJ_TYPE_ID_COM)) {
            interactorType = PSIMITags.TYPE_COMPLEX_SL;
            ArrayList <Element> tempList = XMLUtil.getElement (BINDTags.COMPLEX_ID,
                  BINDInteractorE, XMLUtil.getDepth(BINDInteractorE.getName()));
            if (XMLUtil.validateSingle(BINDTags.COMPLEX_ID, tempList) &&
                XMLUtil.validateInt(tempList.get (0))) {    //only one, if any
                CVMapEntry CVMapped = CVMapEntry.getMappedEntry(PSIMITags.BIND_COMPLEXID_SL);
                refType = CVMapped.getAddInfo(interactorType,
                                              tempList.get (0).getTextTrim ());
                refTypeAccn = PSIMITags.getRefTypeAccn(refType);
                setXRef (interactorE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                         CVMapped.getPSIMITermID (),
                         tempList.get (0).getTextTrim(), refTypeAccn, refType);
                pRefSetFlag = true;
            }
        }

        if (BINDExtRefE != null) {
            otherDBList = XMLUtil.getElement (BINDTags.OTHERDB, BINDInteractorE,
                                   XMLUtil.getDepth(BINDInteractorE.getName()));
            if (XMLUtil.validateExists(BINDTags.OTHERDB, otherDBList)) {
                for (int i = 0; i < otherDBList.size (); i++) {
                    String otherDBName = otherDBList.get (i).getChildText(BINDTags.OTHERDB_NAME);
                    String otherDBID = null;      //either STRP or INTP
                    if (otherDBList.get (i).getChild (BINDTags.OTHERDB_STRP) != null) {
                        otherDBID = getDefText (
                            otherDBList.get (i).getChild(BINDTags.OTHERDB_STRP), NO_VAL);
                        if (otherDBID.equals (NO_VAL)) {
                            otherDBID = null;
                        }
                    }
                    if (otherDBList.get (i).getChild(BINDTags.OTHERDB_INTP) != null) {
                        if (XMLUtil.validateInt (otherDBList.get (i).getChild (BINDTags.OTHERDB_INTP))) {
                            if (otherDBID != null) {            //already set
                                otherDBID += "(" +
                                             otherDBList.get (i).getChildText(BINDTags.OTHERDB_INTP)
                                             + ")";
                                System.err.println ("Bad xref format: " + otherDBName + " " + otherDBID);
                            } else {
                                otherDBID = otherDBList.get (i).getChildText(BINDTags.OTHERDB_INTP);
                            }
                        }
                    }
                    if (otherDBID != null) {              //because of bad input
                        CVMapEntry CVMapped = CVMapEntry.getMappedEntry(otherDBName);
                        refType = CVMapped.getAddInfo(interactorType, otherDBID);
                        refTypeAccn = PSIMITags.getRefTypeAccn (refType);
                        //System.out.println (CVMapped.toString());
                        if (!pRefSetFlag) {
                            setXRef (interactorE, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                                     CVMapped.getPSIMITermID (), otherDBID, 
                                     refTypeAccn, refType);
                            pRefSetFlag = true;
                        } else {
                            setXRef (interactorE, PSIMITags.SREF, CVMapped.getPSIMITerm_SL(),
                                     CVMapped.getPSIMITermID (), otherDBID,
                                     refTypeAccn, refType);
                        }
                    }
                }
            } 
        }

        if (!pRefSetFlag) {
            setXRef(interactorE, PSIMITags.PREF, PSIMITags.DB_PSIMI, PSIMITags.TYPE_UP_ID);
            return;
        }
    }

    protected void addAttribute (Element parent, String attName, String attValue) {

        if (attValue.length () == 0) {
            System.err.println ("Warning: Dummy attribute value for: " + attName);
            return;
        }
        //System.out.println (attName + GenUtil.COLON + attValue);
        Element attListE = parent.getChild (PSIMITags.ATTLIST, this.psins);
        if (attListE == null) {                            //is first att?
            attListE = new Element (PSIMITags.ATTLIST, this.psins);
            parent.addContent (attListE);
        }
        Element attE = new Element (PSIMITags.ATT, this.psins);
        attE.setAttribute(PSIMITags.NAME_ATT, attName);
        attE.setText(attValue);
        attListE.addContent(attE);
    }

    protected void mapXref (Element parent,String interactorType, String sourceType,
                          String sourceID, int interactorSpeciesID) {

        CVMapEntry CVMapped = null;
        String refType, refTypeAccn;
        //make them all generic GIs, since no mapping info prepared
        if (!isMapIDs()) {
           if (sourceType.equals (ExtResource.LIST_GI)) {
               CVMapped = CVMapEntry.getMappedEntry (PSIMITags.ENTREZ_GI_DB);
               refType = CVMapped.getAddInfo(interactorType, sourceID);
               refTypeAccn = PSIMITags.getRefTypeAccn (refType);
               setXRef (parent, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                        CVMapped.getPSIMITermID (), sourceID, refTypeAccn, refType);
           }
           return;
        }
        ArrayList <String> uniprotList,
                  supUniprotList = new ArrayList <String> (),
                  refSeqProList = new ArrayList <String> (),
                  refSeqNucList = new ArrayList <String> (),
                  genProList = new ArrayList <String> ();
        //applies to all source Types
        if (interactorSpeciesID != Integer.parseInt (PSIMITags.TAXID_UN_ID)) {
            uniprotList = IdentifierMapperService.getUniprotIDByXref(sourceType, sourceID,
                        interactorSpeciesID);
        } else {
            uniprotList = IdentifierMapperService.getUniprotIDByXref (sourceType, sourceID);
        }
        //get the GI x-refs based on available mappings
        String sourceTerm = null;
        if (sourceType.equals (ExtResource.LIST_GI)) {
           refSeqProList = IdentifierMapperService.getXrefLO (sourceType, sourceID,
                           ExtResource.LIST_REFSEQ_PRO);
           refSeqNucList = IdentifierMapperService.getXrefLO (sourceType, sourceID,
                           ExtResource.LIST_REFSEQ_RNA);
           genProList = IdentifierMapperService.getXrefLO(sourceType, sourceID,
                           IdentifierMapperService.LO_PROTEIN);
           supUniprotList = IdentifierMapperService.getXrefLO(sourceType, sourceID,
                            IdentifierMapperService.LO_GI_UNI);
           if (uniprotList.size () > 0 || refSeqProList.size() > 0 ||
               genProList.size() > 0) {
               sourceTerm = PSIMITags.ENTREZ_GI__PRO_DB;
           } else if (refSeqNucList.size () > 0) {
               sourceTerm = PSIMITags.ENTREZ_GI_NUC_DB;
           } else {
               sourceTerm = PSIMITags.ENTREZ_GI_DB;
           }
           CVMapped = CVMapEntry.getMappedEntry(sourceTerm);
           refType = CVMapped.getAddInfo(interactorType, sourceID);
           refTypeAccn = PSIMITags.getRefTypeAccn (refType);
           setXRef (parent, PSIMITags.PREF, CVMapped.getPSIMITerm_SL(),
                    CVMapped.getPSIMITermID (), sourceID, refTypeAccn, refType);
        }
        //works for any source type
        if (uniprotList.size() > 0) {
            addMappedRefList (parent, interactorType, sourceID,
                           PSIMITags.BIND_UNIPROT_SL, uniprotList);
        }
        if (supUniprotList.size () > 0) {
            addMappedRefList (parent, interactorType, sourceID,
                              PSIMITags.BIND_UNIPROT_SL, supUniprotList);
        }
        if (refSeqProList.size () > 0) {
            addMappedRefList (parent, interactorType, sourceID,
                              PSIMITags.BIND_REFSEQ_P_SL, refSeqProList);
        }
        if (refSeqNucList.size() > 0) {
            addMappedRefList (parent, interactorType, sourceID,
                              PSIMITags.BIND_REFSEQ_M_SL, refSeqNucList);
        }
        if (genProList.size() > 0) {
            addMappedRefList (parent, interactorType, sourceID,
                              PSIMITags.BIND_PRO_SL, genProList);
        }
    }

    //helper method
    protected void addMappedRefList (Element parent, String interactorType, String
                     sourceID, String listType, ArrayList <String> mappedList) {

        CVMapEntry CVMapped = CVMapEntry.getMappedEntry(listType);
        String refType = CVMapped.getAddInfo(interactorType, sourceID);
        String refTypeAccn = PSIMITags.getRefTypeAccn (refType);
        for (int i = 0; i < mappedList.size (); i++) {
            setXRef (parent, PSIMITags.SREF, CVMapped.getPSIMITerm_SL(),
                    CVMapped.getPSIMITermID (), mappedList.get (i),
                    refTypeAccn, refType);
        }
    }

    //mostly geared for BIND's bad data.
    protected String getDefText (Element element, String defValue) {

        if (element.getText().trim().length () > 0 &&
            !element.getText().trim().equalsIgnoreCase(BINDTags.NULL_VAL) &&
            !element.getText().trim().equals (BINDTags.NULL_VAL2) &&
            !element.getText().trim ().equals (BINDTags.NULL_VAL3) &&
            !element.getText().trim ().contains (BINDTags.NULL_VAL4) &&
            !element.getText().trim ().equalsIgnoreCase(BINDTags.NULL_VAL5) &&
            !element.getText().trim ().equalsIgnoreCase(BINDTags.NULL_VAL6)) {

            return element.getText().trim();
        } else {
           //System.err.println ("Invalid text for element: " +
           //       element.getName() + " " + element.getText().trim());
            
           return defValue;
        }
    }

    protected void addSpecies (int speciesID) {

        if (speciesID > 0) {
            if (!this.BINDSpeciesList.contains(String.valueOf(speciesID))) {
                this.BINDSpeciesList.add(String.valueOf(speciesID));
            }
        }
    }

    private void dumpBINDSpecies () throws Exception {

        for (int i = 0; i < this.BINDSpeciesList.size(); i++) {
            String speciesIDStr = this.BINDSpeciesList.get (i);
            String speciesStr = DWUtil.dumpSpeciesNames(Integer.parseInt(speciesIDStr));
            log.println (speciesIDStr + GenUtil.TAB + speciesStr);
        }
    }
}
