package org.pathwaycommons.trans.util;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.TreeMap;

import org.genemania.dw.util.DefParams;
import org.genemania.dw.util.GenUtil;
import org.pathwaycommons.trans.PSIMITags;

/**
 * Parent class for Basic mining for BIND PSI-MI files
 * Can mine one 'attribute' at a time, for a single PSI-MI file or a
 * whole directory of them.
 * The resutMap is as follows:
 * key: input file name
 * value-key: term/info tracked
 * value-value: count
 *
 * @author rashadbadrawi
 */

public abstract class PSIMIMiner {

    public static final String MINING_INT_DET_METHOD = PSIMITags.INT_DET_METHOD;
    public static final String MINING_INTERACTOR_TYPE = PSIMITags.INT_TYPE;
    public static final String MINING_COMPLEX_SUBUNIT_NUM =
                                              PSIMITags.ATT_COMPLEX_SUBUNIT_CNT;
    public static final String MINING_INT_XREF = PSIMITags.XREF;
    public static final String MINING_BIND_DIV = PSIMITags.ATT_BIND_DIVISION;
    public static final String MINING_METHOD_XML = "MINING_XML";
    public static final String MINING_METHOD_MODEL = "MINING_MODEL";

    protected static final String DEF_REPORT_NAME_PREFIX = "PSIMIMiner";
    protected static final String DEF_REPORT_NAME_SUFFIX = ".txt";
    protected static final String FILE_NAME_HEADER = "File Name";
    protected static final String INT_PROCESSED_HEADER = "Interactions Processed";
    
    protected static PrintWriter log;
    protected int interactionCnt;
    protected int skipCnt;
    protected String reportFileName;
    protected String miningType;

    protected TreeMap <String, TreeMap <String, Integer>> resultMap = new
            TreeMap <String, TreeMap <String, Integer>> ();
    protected TreeMap <String, Integer> termMap;
    protected TreeMap <String, Integer> intCntMap = new TreeMap <String, Integer> ();
    private TreeMap <String, Integer> globalTermMap = new TreeMap <String, Integer> ();

    public static PSIMIMiner getMiner (String inputFileName, String reportFileName,
                  String miningType, String miningMethod) throws Exception {

        if (PSIMIMiner.MINING_METHOD_XML.equals (miningMethod)) {
            return new PSIMIMinerXML (inputFileName, reportFileName, miningType);
        } else if (PSIMIMiner.MINING_METHOD_MODEL.equals (miningMethod)) {
            return new PSIMIMinerModel(inputFileName, reportFileName, miningType);
        } else {
            throw new IllegalArgumentException ("Invalid mining method: " +
                                               miningMethod);
        }
    }

    protected PSIMIMiner (String inputFileName, String reportFileName,
                          String miningType) throws Exception {

        GenUtil.validateString(inputFileName);
        setReportFileName (reportFileName);
        setMiningType (miningType);

        if (new File (inputFileName).isDirectory ()) {
            File filesArr [] = new File (inputFileName).listFiles();
            String currentFileName;
            Arrays.sort (filesArr);
            int cnt = 1;
            for (int i = 0; i < filesArr.length; i++) {
                if (!filesArr [i].getAbsolutePath().endsWith(XMLUtil.XML_FILE_SUFFIX)) {
                    continue;
                }
                currentFileName = filesArr [i].getAbsolutePath();
                System.out.println ("Mining file: " + cnt++ + " " + currentFileName);
                minePSIMI (currentFileName);
           }
        } else {
            minePSIMI (inputFileName);
        }
    }

    private void setReportFileName (String reportFileName) {

        GenUtil.validateString(reportFileName);
        this.reportFileName = reportFileName;
    }

    private void setMiningType (String miningType) {

        GenUtil.validateString(miningType);
        this.miningType = miningType;
    }

    protected void trackTermCnt (String term) {

        //increment for the current file
        if (this.termMap.containsKey (term)) {
            this.termMap.put(term,
            new Integer (this.termMap.get (term).intValue() + 1));
        } else {
            this.termMap.put(term, new Integer (1));
        }
        //increment 'globally'
        if (this.globalTermMap.containsKey (term)) {
            this.globalTermMap.put(term,
            new Integer (this.globalTermMap.get (term).intValue() + 1));
        } else {
            this.globalTermMap.put(term, new Integer (1));
        }

        //System.out.println (term);
    }

    protected void incrementCounts (String inputFileName) {

        if (this.termMap.size() > 0) {
            this.resultMap.put(inputFileName, this.termMap);
        }
        this.intCntMap.put (inputFileName, new Integer (this.interactionCnt));

        System.gc();
        System.out.println ("Total interactions in: " +
                             inputFileName + ": " + this.interactionCnt);
    }

    protected void dump () throws IOException {

        File reportFile = new File (this.reportFileName,
             PSIMIMiner.DEF_REPORT_NAME_PREFIX + GenUtil.UNDERSCORE +
             this.miningType + PSIMIMiner.DEF_REPORT_NAME_SUFFIX);
        BufferedWriter bw = new BufferedWriter (new FileWriter
                                               (reportFile.getAbsolutePath()));
        //log global counts
        Iterator iterator = this.globalTermMap.keySet().iterator();
        while (iterator.hasNext()) {
            String term = (String)iterator.next();
            log.println(term + GenUtil.TAB + this.globalTermMap.get (term));
            System.out.println (term + GenUtil.TAB + this.globalTermMap.get (term));
        }
        log.println();
        //add headers
        ArrayList <String> tempList = new ArrayList <String> ();
        tempList.addAll (this.globalTermMap.keySet());
        bw.write (PSIMIMiner.FILE_NAME_HEADER + GenUtil.TAB +
                  PSIMIMiner.INT_PROCESSED_HEADER + GenUtil.TAB);
        for (int i = 0; i < tempList.size (); i++) {
            bw.write (tempList.get(i) + GenUtil.TAB);
        }
        bw.newLine();
        int totalIntCnt = 0;
        //add details per file
        iterator = this.resultMap.keySet().iterator();
        while (iterator.hasNext ()) {
            String currentFileName = (String)iterator.next ();
            String reportStr = currentFileName + GenUtil.TAB +
               this.intCntMap.get (currentFileName).toString() + GenUtil.TAB;
            totalIntCnt += this.intCntMap.get (currentFileName).intValue();
            for (int i = 0; i < tempList.size (); i++) {
                String cntStr = GenUtil.NA;
                TreeMap <String, Integer> currentTermsMap =
                                          this.resultMap.get (currentFileName);
                Integer cntInt = currentTermsMap.get (tempList.get (i));
                if (cntInt != null) {
                    cntStr = cntInt.toString();
                }
                reportStr += cntStr + GenUtil.TAB;
            }
            //System.out.println (reportStr);
            bw.write (reportStr);
            bw.newLine();
        }
        bw.flush();
        bw.close();
        System.out.println ("Report generated: " + reportFile.getCanonicalPath());
        System.out.println ("Total files processed: " +
                this.resultMap.size() + ". Total terms found/processed: " +
                this.globalTermMap.size() + ".\nTotal interactions: " +
                totalIntCnt + ". Total skipped: " + this.skipCnt);

        this.resultMap.clear();
        this.globalTermMap.clear();
        this.interactionCnt = 0;
        this.skipCnt = 0;
        System.gc();
    }

    protected void mineComplexSubunits () {

        trackAtt (PSIMITags.ATT_COMPLEX_SUBUNIT_CNT);
    }

    protected void mineBINDDivCategory () {

        trackAtt (PSIMITags.ATT_BIND_DIVISION);
    }

    protected abstract void minePSIMI (String inputFileName) throws Exception;

    protected abstract void mineDetMethod ();

    protected abstract void mineInteractorType () throws Exception;

    protected abstract void mineInteractorXRef () throws Exception;
    //helper methods, assumes att is listed once in an att list.
    protected abstract void trackAtt (String attName);
  
    protected abstract String getAtt (String attName);

    public static void main (String args []) throws Exception {

        log = GenUtil.getDefaultLog();
        log.println(GenUtil.getTimeStamp());
        GenUtil.registerStart();
        String usageMsg = "Usage: PSIMIMiner MiningType [" +
             PSIMITags.INT_DET_METHOD + GenUtil.PIPE + PSIMITags.INT_TYPE +
             GenUtil.PIPE + PSIMITags.ATT_COMPLEX_SUBUNIT_CNT + GenUtil.PIPE +
             PSIMITags.XREF + GenUtil.PIPE + PSIMITags.ATT_BIND_DIVISION +
             "] inputpath miningMethod [" + PSIMIMiner.MINING_METHOD_XML +
             "|" + PSIMIMiner.MINING_METHOD_MODEL + "]";
        String warnMsg = "WARNING: Missing command line args, using defaults";
        if (args == null || args.length < 3) {
            log.println(warnMsg);
            log.println(usageMsg);
            System.out.println(warnMsg);
            System.out.println(usageMsg);
            args = DefParams.getCommandLineArgs(PSIMIMiner.class.getName());
        }
        try {
            String reportFileName = DefParams.getDefaultProp(DefParams.TOOLS_PATH_PROP);
            PSIMIMiner miner = getMiner (args [1], reportFileName , args [0], args [2]);
            miner.dump ();
        } catch (Throwable e) {
            e.printStackTrace();
            e.printStackTrace(log);
        } finally {
            log.println(GenUtil.getExecTimeStr());
            log.flush();
            log.close();
        }
   }
}
